/*Discngine Web Panel Javascript API 4.1*/
/*All rights reserved. Discngine S.A.S 2018*/

"use strict";

/** @namespace TIBCO Spotfire Server */
/** @namespace Document Manipulation */
/** @namespace Data Table Manipulation */
/** @namespace Template Manipulation */
/** @namespace Event Management */
/** @namespace Marking */
/** @namespace Highlight */
/** @namespace Filters */
/** @namespace Script */
/** @namespace Utils */
/** @namespace TIBCO Spotfire Visualization */
/** @namespace TIBCO Spotfire Visualization Configuration */
/** @namespace TIBCO Spotfire Visualization Properties - General */
/** @namespace TIBCO Spotfire Visualization Properties - Data */
/** @namespace TIBCO Spotfire Visualization Properties - Appearance */
/** @namespace TIBCO Spotfire Visualization Properties - Fonts */
/** @namespace TIBCO Spotfire Visualization Properties - Axis */
/** @namespace TIBCO Spotfire Visualization Properties - Line By */
/** @namespace TIBCO Spotfire Visualization Properties - Columns */
/** @namespace TIBCO Spotfire Visualization Properties - Series */
/** @namespace TIBCO Spotfire Visualization Properties - Colors */
/** @namespace TIBCO Spotfire Visualization Properties - Size */
/** @namespace TIBCO Spotfire Visualization Properties - Hierarchy */
/** @namespace TIBCO Spotfire Visualization Properties - Shape */
/** @namespace TIBCO Spotfire Visualization Properties - Labels */
/** @namespace TIBCO Spotfire Visualization Properties - Tooltip */
/** @namespace TIBCO Spotfire Visualization Properties - Legend */
/** @namespace TIBCO Spotfire Visualization Properties - Trellis */
/** @namespace TIBCO Spotfire Visualization Properties - Line Connection */
/** @namespace TIBCO Spotfire Visualization Properties - Marker By */
/** @namespace TIBCO Spotfire Visualization Properties - Statistics Table */
/** @namespace TIBCO Spotfire Visualization Properties - Lines & Curves */
/** @namespace TIBCO Spotfire Visualization Properties - Error Bars */

var connectedToSpotfire;
var externalinterface;
var showAlerts;
var initialPage;

try {
    externalinterface = window.external;
    connectedToSpotfire = true;
    showAlerts = false;
    window.status = "Connected to Spotfire";
} catch (error) {
    connectedToSpotfire = false;
    window.status = "Not connected to Spotfire";
    if(showAlerts){
        alert("Error : " + error.message + "\nWarning : " + window.status);
    }
    else{
        throw new Error("Error : " + error.message + "\nWarning : " + window.status);
    }
}

/**
 * Shows alert when an error occured. 
 * 
 * @memberOf Utils
 * @param {boolean} True to show warnings and errors in alerts, False otherwise. 
 */ 
function ShowAlerts(b){
    showAlerts = b;
}

var logger = {
    log: function(p_logMessage) {
        if(showAlerts){
            alert(p_logMessage);
        }
        else{
            throw new Error(p_logMessage);
        }
    },

    logWarning: function(p_warningMessage) {
        this.log("Warning " + p_warningMessage);
    },

    logError: function(p_errorMessage) {
        this.log("Error " + p_errorMessage);
    }
};

function checkExternalInterface(p_functionName) {
    if ( externalinterface === null ) {
        logger.logWarning(p_functionName + ": " + window.status);
        return false;
    }

    return true;
}

function checkWrapper(p_functionName, p_functionCallback) {
    if( checkExternalInterface(p_functionName) ) {
        try {
            p_functionCallback();
        }
        catch(error){
            logger.logError(p_functionName + ": " + error.message);
        }
    }
}




/**
 * Gets the absolute URI of the TIBCO Spotfire server.
 *
 * @memberOf TIBCO Spotfire Server
 * @return {String} The absolute URI of the TIBCO Spotfire server.
 */
function getServerUri(){
    var uri = "";
    checkWrapper("getServerUri", function () {
        uri = externalinterface.getServerUri();
    });
    return uri;
}

/*************INTERNAL*PROPERTIES*METHODS**********/

/**
 * Gets error message generated by the Discngine Web Panel.
 *
 * @memberOf Utils
 * @return {string} The message error.
 */
function getError() {
    var errormsg = "";
    checkWrapper("getError", function () {
        errormsg = externalinterface.LastError;
    });
    return errormsg;
}

/**
 * Checks if the user is connected to a TIBCO Spotfire server.
 *
 * @memberOf TIBCO Spotfire Server
 * @return {boolean} True if the server is on-line, false otherwise.
 */
function IsServerOnline(){
    var online = false;
    checkWrapper("IsServerOnline", function () {
        online = externalinterface.IsOnline();
        if(online == false) {
            var errormsg = getError();
            if(errormsg != "") {
                throw new Error(errormsg);
            }
        }
    });
    return online;
}

/**
 * Gets the active TIBCO Spotfire page name.
 *
 * @memberOf Document Manipulation
 * @return {string} The page name.
 */
function getOwnerPageTitle() {
    var title = "";
    checkWrapper("getOwnerPageTitle", function () {
        title = externalinterface.PageTitle;
    });
    return title;
}

/**
 * Gets the active Delimiter to delimit all lists.
 *
 * @memberOf Utils
 * @return {string} The active delimiter.
 */
function getDelimiter() {
    var delimiter = "";
    checkWrapper("getDelimiter", function () {
        delimiter = externalinterface.Delim;
    });
    return delimiter;
}

/**
 * Sets the active Delimiter to delimit all lists, use this to put a delimiter for all the session.
 *
 * @memberOf Utils
 * @param {string} delimiter The active delimiter to set.
 */
function setDelimiter(delimiter) {
    checkWrapper("setDelimiter", function () {
        externalinterface.Delim = delimiter;
    });
}

/**
 * Gets the name of the id column of the active TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @return {string} The name of the id column of the active TIBCO Spotfire Data Table.
 */
function getIdColumn() {
    var columnname = "";
    checkWrapper("getIdColumn", function () {
        columnname = externalinterface.IdColumn;
    });
    return columnname;
}

/**
 * Sets the name of the id column of the active TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} columnname The name of the id column of the active TIBCO Spotfire Data Table to set.
 */
function setIdColumn(columnname) {
    checkWrapper("setIdColumn", function () {
        externalinterface.IdColumn = columnname;
        var errormsg = getError();
        if(errormsg != "") {
            throw new Error(errormsg);
        }
    });
}


/**
 * Gets the name of the active Marking.
 *
 * @memberOf Marking
 * @return {string} The name of the active Marking.
 */
function getMarking() {
    var marking = "";
    checkWrapper("getMarking", function () {
        marking = externalinterface.Marking;
    });
    return marking;
}


/**
 * Sets the name of the active Marking.
 *
 * @memberOf Marking
 * @param {string} marking The name of the active Marking to set.
 * @exception Message error, use getError method to get the error message.
 */
function setMarking(marking) {
    checkWrapper("setMarking", function () {
        externalinterface.Marking = marking;
        var errormsg = getError();
        if(errormsg != "") {
            throw new Error(errormsg);
        }
    });
}

/*************DOCUMENT*METHODS******************************************************************************************************/

/**
 * Gets the value of the specified document property.
 *
 * @memberOf Document Manipulation
 * @param {string} propertyname The name of the document property.
 * @return {string} The value of the document property.
 * @exception If the property cannot be found in the document.
 */
function getDocumentProperty(propertyname) {
    var value = null;
    checkWrapper("getDocumentProperty", function () {
        value = externalinterface.GetDataProperty(propertyname);
    });
    return value;
}

/**
 * Sets a value to the specified document property. If the document property does not exist, it is created and the provided value is setted. 
 *
 * @memberOf Document Manipulation
 * @param {string} propertyname The name of the document property. If it does not exist, it is created. 
 * @param {string} propertyvalue The value of the document property to set.
 */
function setDocumentProperty(propertyname, propertyvalue) {
    checkWrapper("setDocumentProperty", function () {
        if (externalinterface.DocumentOpen) {
            var dummy = externalinterface.SetDataProperty(propertyname, propertyvalue);
            if(!dummy) {
                throw new Error(getError());
            }
        }
        else {
            logger.logError("setDocumentProperty: no Document opened to set a property.");
        }
    });
}

/**
 * Gets the document property names list.
 *
 * @memberOf Document Manipulation
 * @param {string} separator The separator to use to separate property names.
 * @return {string} The document property names separated by the provided separator.
 */
function getDocumentPropertyNames(separator) {
    var value = null;
    checkWrapper("getDocumentPropertyNames", function () {
        value = externalinterface.GetDataPropertyNames(separator);
    });
    return value;
}

/**
 * Saves the current TIBCO Spotfire document in the given TIBCO Spotfire Libray folder
 *
 * @memberOf Document Manipulation
 * @param {string} webpanelurl The url of Discngine Web Panel to set before saving document (it will be browsed when the document is reopened)
 * @param {string} documenttitle The name of document which it will be saved into
 * @param {string} libraryfoldername The name of the TIBCO Spotfire Library folder where the document is saved. The folder must exist.
 * @param {string} createfolderifmissing Optional boolean parameter defaulted to false, when true the function will try to create the folder if it did not exist and the user have the rights to do so.
 * @exception If the TIBCO Spotfire Library folder cannot be reached. 
 */
function saveDocumentOnLibrary(webpanelurl, documenttitle, libraryfoldername, createfolderifmissing){
    checkWrapper("saveDocumentOnLibrary", function () {
        //1-Set current url on Document property
        if(webpanelurl =='' || webpanelurl == null) {
            webpanelurl = window.location.href;
        }
        setDocumentProperty("DiscngineWebPanelURL",webpanelurl);
        //2-Save document on server library
        if (typeof createfolderifmissing === 'undefined') { createfolderifmissing = 'false'; }
        if (typeof createfolderifmissing === 'boolean') { createfolderifmissing = createfolderifmissing.toString(); }
            
        var b = externalinterface.SaveDocumentOnLibrary(documenttitle, libraryfoldername, createfolderifmissing);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Saves the current TIBCO Spotfire document in the given path.
 *
 * @memberOf Document Manipulation
 * @param {string} webpanelurl The url of Discngine Web Panel to set before saving document (it will be browsed when the document is reopened).
 * @param {string} documentpath The path which the document is saved into.
 */
function saveDocumentOnLocal(webpanelurl, documentpath){
    checkWrapper("saveDocumentOnLocal", function () {
        //1-Set current url on Document property
        if(webpanelurl =='' || webpanelurl == null) {
            webpanelurl = window.location.href;
        }
        setDocumentProperty("DiscngineWebPanelURL",webpanelurl);
        //2-Save document on server library
        var b = externalinterface.SaveDocumentOnLocal(documentpath);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Saves the current TIBCO Spotfire document as a template in the given path.
 * A template is a TIBCO Spotfire document which contains a description of tables, visualisations... but does not contains data.
 *
 * @memberOf Template Manipulation
 * @param {string} webpanelurl The url of Discngine Web Panel to set before saving template (it will be browsed when the template is applied).
 * Left this parameter empty to avoid the Web Panel redirection.
 * @param {string} templatepath The path which the template is saved into (without data).
 */
function saveTemplateToFile(webpanelurl, templatepath){
    checkWrapper("saveTemplateToFile", function () {
        var b;
        if (webpanelurl != null && webpanelurl != "") {
            setDocumentProperty("DiscngineWebPanelURL",webpanelurl);
            b = externalinterface.SaveTemplateToFile(templatepath, true);
        }
        else {
            b = externalinterface.SaveTemplateToFile(templatepath, false);
        }
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Saves the current TIBCO Spotfire document as a template in the given TIBCO Spotfire Library folder.
 * A template is a TIBCO Spotfire document which contains a description of tables, visualisations... but does not contain data.
 *
 * @memberOf Template Manipulation
 * @param {string} webpanelurl The url of Discngine Web Panel to set before saving template (it will be browsed when the template is applied).
 * Left this parameter empty to avoid the Web Panel redirection.
 * @param {string} templatetitle The path which the document is saved into.
 * @param {string} libraryfoldername The name of the TIBCO Spotfire Library folder. 
 * @param {string} createfolderifmissing Optional boolean parameter defaulted to false, when true the function will try to create the folder if it did not exist and the user have the rights to do so.
 * @exception If the TIBCO Spotfire Library folder cannot be reached. 
 */
function saveTemplateToLibrary(webpanelurl, templatetitle, libraryfoldername, createfolderifmissing){
    checkWrapper("saveTemplateToLibrary", function () {
        var b;
        // Check if createfolderifmissing is set.
        if (typeof createfolderifmissing === 'undefined') { createfolderifmissing = 'false'; }
        if (typeof createfolderifmissing === 'boolean') { createfolderifmissing = createfolderifmissing.toString(); }
        if (webpanelurl != null && webpanelurl != "") {
            setDocumentProperty("DiscngineWebPanelURL",webpanelurl);
            b = externalinterface.SaveTemplateToLibrary(templatetitle,  libraryfoldername, true, createfolderifmissing);
        }
        else {
            b = externalinterface.SaveTemplateToLibrary(templatetitle, libraryfoldername, false, createfolderifmissing);
        }
        if(!b) {
            throw new Error(getError());
        }
    });
}

 /**
 * Applies  a template loaded from TIBCO Spotfire Library to the current TIBCO Spotfire document.
 * A template is a TIBCO Spotfire document which contains a description of tables, visualisations... but does not contains data.
 * The current document is saved.
 *
 * @memberOf Template Manipulation
 * @param {string} webpanelurl The url of Discngine Web Panel to set after applying template.
 * Left this parameter empty to avoid the Web Panel redirection.
 * @param {string} templatetitle The name of the template to apply.
 * @param {string} documenttitle The name which the document is saved into. If null, the document is not saved and
 * modifications of the document will be saved in the template.
 * @param {string} libraryfoldername The name of the library folder.
 * @param {string} localfolder The path of a local folder where some settings are saved temporary.
 * @exception If the TIBCO Spotfire Library folder cannot be reached. 
 * @example applyTemplateFromLibrary('http://server/DiscngineWebpanel/', 'test_template.dxp', 'test_template_data.dxp', 'Discngine Library', 'c:/temp/'); // save the document
 * @example applyTemplateFromLibrary('http://server/DiscngineWebpanel/', 'test_template.dxp', '', 'Discngine Library', 'c:/temp/'); // do not save the document
 * @example applyTemplateFromLibrary('http://server/DiscngineWebpanel/', 'test_template.dxp', 'null', 'Discngine Library', 'c:/temp/'); // do not save the document
 * @example applyTemplateFromLibrary('http://server/DiscngineWebpanel/', 'test_template.dxp', null, 'Discngine Library', 'c:/temp/'); // do not save the document
 * @example applyTemplateFromLibrary('', 'test_template.dxp', null, 'Discngine Library', 'c:/temp/'); // do not redirect the Web Panel
 */
function applyTemplateFromLibrary(webpanelurl, templatetitle, documenttitle, libraryfoldername, localfolder){
    checkWrapper("applyTemplateFromLibrary", function () {
        var b;
        if (webpanelurl != null && webpanelurl != "") {
            setDocumentProperty("DiscngineWebPanelURL",webpanelurl);
            b = externalinterface.applyTemplateFromLibrary(templatetitle, documenttitle, libraryfoldername, localfolder, true);
        }
        else {
            b = externalinterface.applyTemplateFromLibrary(templatetitle, documenttitle, libraryfoldername, localfolder, false);
        }
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Applies  a template loaded from a path to the current TIBCO Spotfire document.
 * A template is a TIBCO Spotfire document which contains a description of tables, visualisations... but does not contains data.
 * The current document is saved.
 *
 * @memberOf Template Manipulation
 * @param {string} webpanelurl The path of Discngine Web Panel to set after applying template.
 * Left this parameter empty to avoid the Web Panel redirection.
 * @param {string} templatepath The path of the template to apply.
 * @param {string} documentpath The path which the document is saved into. If null, the document is not saved and
 * modifications of the document will be saved in the template.
 * @example applyTemplateFromLocal('http://server:port/DiscngineWebpanel/', 'c:/temp/test_template.dxp', 'c:/temp/test_template_data.dxp'); // save the document
 * @example applyTemplateFromLocal('http://server:port/DiscngineWebpanel/', 'c:/temp/test_template.dxp', ''); // do not save the document
 * @example applyTemplateFromLocal('http://server:port/DiscngineWebpanel/', 'c:/temp/test_template.dxp', 'null'); // do not save the document
 * @example applyTemplateFromLocal('http://server:port/DiscngineWebpanel/', 'c:/temp/test_template.dxp', null); // do not save the document
 * @example applyTemplateFromLocal('', 'c:/temp/test_template.dxp', null); // do not redirect the Web Panel
 */
function applyTemplateFromLocal(webpanelurl, templatepath, documentpath){
    checkWrapper("applyTemplateFromLocal", function () {
        var b;
        if (webpanelurl != null && webpanelurl != "") {
            setDocumentProperty("DiscngineWebPanelURL",webpanelurl);
            b = externalinterface.ApplyTemplateFromLocal(templatepath, documentpath, true);
        }
        else {
            b = externalinterface.ApplyTemplateFromLocal(templatepath, documentpath, false);
        }
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Adds a new page to the current TIBCO Spotfire Document
 *
 * @memberOf Document Manipulation
 * @param {string} title The title of the newly created page.
 * @param {boolean} replaceifexists If true, an existing page with the name "title" will be replaced. If false, the added page will have an indexed title.
 * @param {boolean} setAsActive Set new page as active page? Default is true.
 */
function addPage(title, replaceifexists, setAsActive)
{
    checkWrapper("addPage", function () {
        var b;
        if (typeof setAsActive !== "undefined") {
            b = externalinterface.AddPage(title, replaceifexists, setAsActive);
        }
        else {
            b = externalinterface.AddPage(title, replaceifexists, "true");
        }
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Deletes a TIBCO Spotfire Document from TIBCO Spotfire Library.
 *
 * @memberOf Document Manipulation
 * @param {string} documenttitle The name of the document to delete.
 */
function deleteFileFromLibrary(documenttitle){
    checkWrapper("deleteFileFromLibrary", function () {
        var b = externalinterface.DeleteFileFromLibrary(documenttitle);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Opens a TIBCO Spotfire Document from a local file.
 *
 * @memberOf Document Manipulation
 * @param {string} filepath Document file location.
 * @exception If the local file cannot be reached. 
 * @example openDocumentFromLocalFile("directory\\document.dxp");
 */
function openDocumentFromLocalFile(filepath)
{
    checkWrapper("openDocumentFromLocalFile", function () {
        var b = externalinterface.OpenDocumentFromLocalFile(filepath);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Opens a TIBCO Spotfire Document from Library.
 *
 * @memberOf Document Manipulation
 * @param {string} librarypath Document file location in TIBCO Spotfire Library.
 * @exception If the TIBCO SPotfire Document cannot be reached. 
 */
function openDocumentFromLibrary(librarypath)
{
    checkWrapper("openDocumentFromLibrary", function () {
        var b = externalinterface.OpenDocumentFromLibrary(librarypath);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Opens a TIBCO Spotfire Document from Library with parameters.
 *
 * @memberOf Document Manipulation
 * @param {string} librarypath Document file location in TIBCO Spotfire Library.
 * @param {string} parameters Configuration block text that shall be used to perform an initial configuration of the document after it has been opened.
 * The characters "=", ";", "{", "}", "," have to be escaped by "\\" in a parameter name or in a value.
 * @exception If the TIBCO SPotfire Document cannot be reached. 
 * @example openDocumentFromLibraryWithParameters("/myDirectoryInLibrary/myDocument", "val_run_id=1;");
 */
function openDocumentFromLibraryWithParameters(librarypath, parameters)
{
    checkWrapper("openDocumentFromLibraryWithParameters", function () {
        var b = externalinterface.OpenDocumentFromLibraryWithParameters(librarypath, parameters);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Shows or hides the Filters panel.
 *
 * @memberOf Document Manipulation
 * @param {boolean} show True to show the Filters panel, False to hide it.
 * @exemple showFiltersPanel('False');
 */
function showFiltersPanel(show)
{
    checkWrapper("showFiltersPanel", function () {
        var b = externalinterface.ShowFiltersPanel(show);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Shows or hides the Details-on-Demand panel.
 *
 * @memberOf Document Manipulation
 * @param {boolean} show True to show the Details-on-Demand panel, False to hide it.
 * @exemple showDetailsOnDemandPanel('False');
 */
function showDetailsOnDemandPanel(show)
{
    checkWrapper("showDetailsOnDemandPanel", function () {
        var b = externalinterface.ShowDetailsOnDemandPanel(show);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/*************DATA*METHODS******************************************************************************************************/

/**
 * Loads a TIBCO Spotfire Data table from an url.
 *
 * @memberOf Data Table Manipulation
 * @param {string} js_url The url of the file to load.
 * @param {string} datatablename The name of the data table to create.
 * @param {string} keepexistingdatatable If true and "datatablename" already exists in the document, the new datatable will have an indexed name.
 * If false, the existing "datatablename" will be replaced by this one.
 * @exception If the data table name is null or empty.
 */
function loadDataUrl(js_url, datatablename, keepexistingdatatable) {
    checkWrapper("loadDataUrl", function () {
        var b;
        if (typeof keepexistingdatatable !== "undefined") {
            b = externalinterface.AddDataTableFromURL(js_url, datatablename, keepexistingdatatable);
        }
        else {
            b = externalinterface.AddDataTableFromURL(js_url, datatablename, "false");
        }
        if(!b) {
            throw new Error(getError());
        }
        setDocumentProperty("DiscngineWebPanelURL", document.location.href);
    });
}

/**
 * Loads a TIBCO Spotfire Data table from an information link.
 *
 * @memberOf Data Table Manipulation
 * @param {string} informationlinkname The name of the information link to load. If the library is specified, the path must contain "/".
 * @param {string} datatablename The name of the data table to create.
 * @param {string} parameters The parameters string (i.e. "ParameterName=value;" for a single string value, or "ParameterName={value1,value2};" for an array, or
 * "ParameterName=0;" for a numeric value, or "ParameterName1=value1;ParameterName2={value2,value3};" for several parameters).
 * The characters "=", ";", "{", "}", "," have to be escaped by "\\" in a parameter name or in a value.
 * @param {string} keepexistingdatatable If true and "datatablename" already exists in the document, the new datatable will have an indexed name.
 * If false, the existing "datatablename" will be replaced by this one.
 * @exception If the information link cannot be found. 
 * @example addDataTableFromInformationLink("/myLibrary/myDirectory/myInformationLink", "Pipeline Pilot Data 1", "id=idnumber;", "true");
 *  addDataTableFromInformationLink("/myLibrary/myDirectory/myInformationLink", "Pipeline Pilot Data 1",
 *      "id=idnumber;val=a\\,value\\{with\\=escaped\\}characters\\;;", "true");
 */
function addDataTableFromInformationLink(informationlinkname, datatablename, parameters, keepexistingdatatable) {
    checkWrapper("addDataTableFromInformationLink", function () {
        var b;
        if (typeof keepexistingdatatable !== "undefined") {
            b = externalinterface.AddDataTableFromInformationLink(informationlinkname, datatablename, parameters, keepexistingdatatable);
        }
        else {
            b = externalinterface.AddDataTableFromInformationLink(informationlinkname, datatablename, parameters, "false");
        }
        if(!b) {
            throw new Error(getError());
        }
        setDocumentProperty("DiscngineWebPanelURL", document.location.href);
    });
}

/**
 * Adds a TIBCO Spotfire Data columns from an url to a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} js_url The url of the file to load.
 * @param {string} datatablename The name of the data table to which the columns are added. If empty, the Spotfire active data table is used.
 * @param {string} js_matchCol The name of the matching identifier data column.
 * @param {string} js_deleteCols The list of data columns to replace or to add. The names are separated by a delimiter (defined in js_delimiter parameter).
 * @param {string} js_delimiter The delimiter of the columns names list.
 * @exception If the data table does not exist in the document. 
 * @exception If data cannot be getted from the file. 
 * @exception If the matching data column cannot be getted from the data table. 
 * @exception If the matching data column cannot be getted from the file. 
 */
function addDataFromFile(js_url, datatablename, js_matchCol, js_deleteCols, js_delimiter) {
    checkWrapper("addDataFromFile", function () {
        var b = externalinterface.DeleteColumns(datatablename, js_deleteCols, js_delimiter);
        if (!b) {
            throw new Error(getError());
        }
        b = externalinterface.AddColumnFromURL(datatablename, js_url, js_matchCol, js_deleteCols, js_delimiter);
        if (!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Adds a TIBCO Spotfire Data columns from an url to a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of the data table to which the columns are added. If empty, the Spotfire active data table is used.
 * @param {string} informationlinkname The name of the information link to load. If the library is specified, the path must contain "/".
 * @param {string} js_matchCol The name of the matching identifier data column.
 * @param {string} js_deleteCols The list of data columns to replace or to add. The names are separated by a delimiter (defined in js_delimiter parameter).
 * @param {string} js_delimiter The delimiter of the columns names list.
 * @param {string} parameters The parameters string (i.e. ParameterName=value; for a single string value, or ParameterName={value1,value2}; for an array, or
 * ParameterName=0; for a numeric value, or ParameterName1=value1;ParameterName2={value2,value3}; for several parameters).
 * The characters "=", ";", "{", "}", "," have to be escaped by "\\" in a parameter name or in a value.
 * @exception If the data table does not exist in the document. 
 * @exception If the information link is not found in the library. 
 * @exception If the matching data column cannot be getted from the data table. 
 * @exception If the matching data column cannot be getted from the information link. 
 * @example addDataFromInformationLink("Pipeline Pilot Data 1", "/myLibrary/myDirectory/myInformationLink", "IDNUMBER", "col1;col2", ";", "id=idnumber;");
 *  addDataFromInformationLink("Pipeline Pilot Data 1", "/myLibrary/myDirectory/myInformationLink", "IDNUMBER", "col1;col2", ";",
 *      "id=idnumber;val=a\\,value\\{with\\=escaped\\}characters\\;;");
 */
function addDataFromInformationLink(datatablename, informationlinkname, js_matchCol, js_deleteCols, js_delimiter, parameters) {
    checkWrapper("addDataFromInformationLink", function () {
        var b = externalinterface.DeleteColumns(datatablename, js_deleteCols, js_delimiter);
        if(!b) {
            throw new Error(getError());
        }
        b = externalinterface.AddColumnFromInformationLink(datatablename, informationlinkname, js_matchCol, js_deleteCols, js_delimiter, parameters);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Adds a TIBCO Spotfire Data Column from an existing TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} sourcedatatablename The name of the source data table. If empty, the Spotfire active data table is used.
 * @param {string} destinationdatatablename The name of the destination data table.
 * @param {string} js_matchCol_source The name of the matching identifier data column in the source data table.
 * @param {string} js_matchCol_destination The name of the matching identifier data column in the destination data table.
 * @param {string} js_deleteCols The list of data columns to replace or to add. The names are sperated by a delimiter (defined in js_delimiter parameter).
 * @param {string} js_delimiter The delimiter of the columns names list.
 * @exception If the source data table does not exist in the Document. 
 * @exception If the destination data table does not exist in the Document. 
 * @exception If the matching data column cannot be getted from the source data table. 
 * @exception If the matching data column cannot be getted from the destination data table. 
 */
function addDataFromDataTable(sourcedatatablename, destinationdatatablename, js_matchCol_source, js_matchCol_destination, js_deleteCols, js_delimiter ) {
    checkWrapper("addDataFromDataTable", function () {
        var b = externalinterface.addColumnFromDataTable(sourcedatatablename, destinationdatatablename, js_matchCol_source, js_matchCol_destination, js_deleteCols, js_delimiter);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Renames a TIBCO Spotfire Data column from a TIBCO Spotfire Data Table. If the new name is already used in the data table, it is indexed. 
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of the data table from which the column is renamed. If empty, the Spotfire active data table is used.
 * @param {string} js_oldCol The old name of the data column to rename.
 * @param {string} js_newCol The new name of the data column to rename.
 * @exception If the data table does not exist in the Document. 
 */
function renameColumn (datatablename, js_oldCol, js_newCol) {
    checkWrapper("renameColumn", function () {
        var b = externalinterface.RenameColumn(datatablename, js_oldCol, js_newCol);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Removes a TIBCO Spotfire Data column from a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of the data table from which the column is deleted.
 * @param {string} js_deleteCol The name of the data column to delete.
 * @exception If the data table does not exist in the Document. 
 */
function deleteColumn (datatablename, js_deleteCol) {
    checkWrapper("deleteColumn", function () {
        var b = externalinterface.DeleteColumn(datatablename, js_deleteCol);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Removes TIBCO Spotfire Data columns from a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of the data table from which the column is deleted. If empty, the Spotfire active data table is used.
 * @param {string} js_deleteCols The list of the data columns to delete. The columns names are sperated by a delimiter (defined in js_delimiter parameter).
 * @param {string} js_delimiter The delimiter of the columns names list.
 * @exception If the data table does not exist in the Document. 
 */
function deleteColumns (datatablename, js_deleteCols, js_delimiter) {
    checkWrapper("deleteColumns", function () {
        var b = externalinterface.DeleteColumns(datatablename, js_deleteCols, js_delimiter);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Sets the renderer of a TIBCO Spotfire Data Table column.
 *
 * @memberOf Data Table Manipulation
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} column The name of the Data Table column.
 * @param {integer} renderer The id of the built-in renderer : 0 = BitmapRenderer, 1 = DefaultRenderer, 2 = GeometryRenderer, 3 = ImageFromUrlRenderer, 4 = LinkRenderer, 5 = VirtualValueRenderer.
 */
function setcolumnrenderer(id, column, renderer) {
    var b;
    checkWrapper("setcolumnrenderer", function () {
        b = externalinterface.SetColumnRenderer(id, column, renderer);
    });
    return b;
}

/**
 * Marks TIBCO Spotfire Data records matching a specific value. Existing marking is removed.
 *
 * @memberOf Marking
 * @param {string} datatable The name of the data table to mark.
 * @param {string} marking The name of the existing Spotfire marking used to mark data. If empty, the active marking is used.
 * @param {string} column The name of the column which holds the values to mark.
 * @param {string} values The list of values to match. The values are separated by a delimiter (defined in delimiter parameter).
 * @param {string} delimiter The delimiter of the values list.
 * @exception If the data table does not exist in the Document. 
 * @exception If the column does not exist in the data table.
 * @exception If the marking does not exist in the document.
 * @example markRecordsExclusive('Pipeline Pilot Data 1', 'Marking', 'Cluster', '5;6', ';');
 */
function markRecordsExclusive(datatable, marking, column, values, delimiter) {
    checkWrapper("markRecordsExclusive", function () {
        var dummy = externalinterface.MarkData(datatable, column, marking, values, delimiter, "true");
        if(!dummy) {
            throw new Error(getError());
        }
    });
}

/**
 * Marks TIBCO Spotfire Data records matching a specific value. Existing marking is kept.
 *
 * @memberOf Marking
 * @param {string} datatable The name of the data table to mark.
 * @param {string} marking The name of the existing Spotfire marking used to mark data. If empty, the active marking is used.
 * @param {string} column The name of the column which holds the values to mark.
 * @param {string} values The list of values to match. The values are separated by a delimiter (defined in delimiter parameter).
 * @param {string} delimiter The delimiter of the values list.
 * @exception If the data table does not exist in the Document. 
 * @exception If the column does not exist in the data table.
 * @exception If the marking does not exist in the document.
 * @example markRecords('Pipeline Pilot Data 1', 'Marking', 'Cluster', '5;6', ';');
 */
function markRecords(datatable, marking, column, values, delimiter) {
    checkWrapper("markRecords", function () {
        var dummy = externalinterface.MarkData(datatable, column, marking, values, delimiter, "false");
        if(!dummy) {
            throw new Error(getError());
        }
    });
}

/**
 * Returns the number of TIBCO Spotfire Data marked records.
 *
 * @memberOf Marking
 * @param {string} datatable The name of the data table. If empty, the active data table is used.
 * @param {string} marking The name of the marking. If empty, the active marking is used.
 * @return {string} The number of marked records.
 * @exception If the data table does not exist in the Document. 
 * @exception If the marking does not exist in the document.
 */
function getMarkedCount(datatable, marking){
    var count = 0;
    checkWrapper("getMarkedCount", function () {
        count = externalinterface.GetMarkedRecordCount(datatable, marking);
    });
    return count;
}

/**
 * Returns the list of TIBCO Spotfire Data marked records, without duplicate values.
 *
 * @memberOf Marking
 * @param {string} datatable The name of the data table. If empty, the active data table is used.
 * @param {string} marking The name of the marking. If empty, the active marking is used.
 * @param {string} column The name of the column which holds the values to return.
 * @param {string} delimiter The delimiter of the returned values list.
 * @param {integer} maximum number of values read. Beware, only the first "maximum" records are read and the result does not contain duplicate, so "maximum" can be different from the number of values returned. When null all data will be send back.
 * @return {string} The list of the marked records values without duplicate.
 * @exception If the data table does not exist in the Document. 
 * @exception If the column does not exist in the data table.
 * @exception If the marking does not exist in the document.
 */
function getMarkedValues(datatable, marking, column, delimiter, maximum){
    var values = "";
    checkWrapper("getMarkedValues", function () {
        values = externalinterface.GetMarkedRecordValues(datatable, column, marking, delimiter, maximum);
        var errormsg = getError();
        if(values == "" && errormsg != "") {
            throw new Error(errormsg);
        }
    });
    return values;
}

/**
 * Returns TIBCO Spotfire Data highlighted records.
 *
 * @memberOf Highlight
 * @param {string} datatable The name of the data table.
 * @param {string} column The name of the column which holds the values to return.
 * @param {string} delimiter The delimiter of the returned values list.
 * @param {integer} maximum number of value read. When null all data will be send back. Beware maximum can be different from the number of value returned as the result contains no duplicate.
 * @return {string} The list of the highlighted records values.
 * @exception If the data table does not exist in the Document. 
 * @exception If the column does not exist in the data table.
 */
function getHighlightedValues(datatable, column, delimiter, maximum){
    var values = "";
    checkWrapper("getHighlightedValues", function () {
        values = externalinterface.GetHighlightRecordValues(datatable, column, delimiter, maximum);
        var errormsg = getError();
        if(values == "" && errormsg != "") {
            throw new Error(errormsg);
        }
    });
    return values;
}

/**
 * Adds a marking to the TIBCO Spotfire Document.
 *
 * @memberOf Marking
 * @param {string} markingName Name of the marking. Cannot be null or empty.
 * @param {string} markingColor Color of the marking. Can be empty.
 * @param {string} replaceMarking True if a marking with same name will be replaced.
 * @return {string} Name of the created marking. If the provided marking name is already used
 * in the TIBCO Spotfire Document, a new marking with new name is created or the extisting one is replaced.
 * @exception If the marking name is empty.
 * @example addMarking("OrangeMarking", "Orange", "True");
 * @example addMarking("BlueMarking", "#0F6DF1", "False");
 */
function addMarking(markingName, markingColor, replaceMarking)
{
    var marking = "";
    checkWrapper("addMarking", function () {
        marking = externalinterface.AddMarking(markingName, markingColor, replaceMarking);
        var errormsg = getError();
        if(marking == "" && errormsg != "") {
            throw new Error(errormsg);
        }
    });
    return marking;
}

/**
 * Gets the active TIBCO Spotfire Marking name.
 *
 * @memberOf Marking
 * @return {string} The name of the active TIBCO Spotfire Marking.
 */
function getActiveMarking() {
    var b;
    checkWrapper("getActiveMarking", function () {
        b = externalinterface.GetActiveMarking();
        var errormsg = getError();
        if (errormsg != "") {
            throw new Error(errormsg);
        }
    });
    return b;
}

/**
 * Gets the default TIBCO Spotfire Marking name.
 *
 * @memberOf Marking
 * @return {string} The name of the default TIBCO Spotfire Marking.
 */
function getDefaultMarking() {
    var b;
    checkWrapper("getDefaultMarking", function () {
        b = externalinterface.GetDefaultMarking();
        var errormsg = getError();
        if (errormsg != "") {
            throw new Error(errormsg);
        }
    });
    return b;
}

/**
 * Sets a marking as default in the TIBCO SPotfire Document.
 *
 * @memberOf Marking
 * @param {string} markingName Name of the marking.
 * @exception If the marking does not exist in the document. 
 * @example setMarkingAsDefault("BlueMarking");
 */
function setDefaultMarking(markingName)
{
    checkWrapper("setDefaultMarking", function () {
        var dummy = externalinterface.SetDefaultMarking(markingName);
        if(!dummy) {
            throw new Error(getError());
        }
    });
}

/**
 * Returns the first TIBCO Spotfire Data record row index.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of the data table.
 * @param {string} columnname The name of the column which holds the value.
 * @param {string} value The reached value.
 * @return {string} The index of row holding the value, -1 if not found.
 * @exception If the data table does not exist in the document.
 * @exception If the value does not exist in the column. 
 */
function getRowIndexOf(datatablename, columnname, value){
    var index = -1;
    checkWrapper("getRowIndexOf", function () {
        index = externalinterface.GetRowIndex(datatablename, columnname, value);
        if(index == -1) {
            throw new Error(getError());
        }
    });
    return index;
}

/**
 * Returns the TIBCO Spotfire Data record value at a row index.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of the data table.
 * @param {string} columnname The name of the column which holds the value.
 * @param {string} rowindex The index of the row holding the value.
 * @return {string} The value at the index.
 * @exception If the data table does not exist in the document.
 * @exception If the column does not exist in the data table.
 * @exception If the row index does not exist in the data table. 
 */
function getValueAt(datatablename, columnname, rowindex){
    var value = "";
    checkWrapper("getValueAt", function () {
        value = externalinterface.GetValue(datatablename, columnname, rowindex);
        var errormsg = getError();
        if(errormsg!= "") {
            throw new Error(errormsg);
        }
    });
    return value;
}

/**
 * Returns the TIBCO Spotfire Data Tables list of the current TIBCO Spotfire document.
 *
 * @memberOf Data Table Manipulation
 * @param {string} delimiter The delimiter of the returned list of data tables names.
 * @return {string} The list of data tables names.
 */
function getDataTables(delimiter){
    var tables = "";
    checkWrapper("getDataTables", function () {
        tables = externalinterface.GetDataTables(delimiter);
        var errormsg = getError();
        if(errormsg != "") {
            throw new Error(errormsg);
        }
    });
    return tables;
}

/**
 * Returns the columns names of a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of data table.
 * @param {string} delimiter The delimiter of the returned list of columns names.
 * @return {string} The list of data columns names.
 * @exception If the data table does not exist in the document.
 */
function getColumns(datatablename, delimiter){
    var columns = "";
    checkWrapper("getColumns", function () {
        columns = externalinterface.GetColumns(datatablename, delimiter);
        var errormsg = getError();
        if(errormsg != "") {
            throw new Error(errormsg);
        }
    });
    return columns;
}

/**
 * Removes all TIBCO Spotfire Data Tables from the current TIBCO Spotfire Document.
 *
 * @memberOf Data Table Manipulation
 * @example deleteAllDataTables();
 */
function deleteAllDataTables(){
    checkWrapper("deleteAllDataTables", function () {
        var b = externalinterface.RemoveAllDataTables();
        if(!b) {
            throw new Error(getError());
        }
    });
    return b;
}

/**
 * Removes TIBCO Spotfire Data Tables from the current TIBCO Spotfire Document.
 * If one of datatables to remove is implicated into a relation at the left
 * position and the right datatable of the relation is not in the list,
 * no datatable is removed and an error message is displayed.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatables The list of data tables names to deleted, separated by a comma.
 * @exception If a data table to remove is implicated in a relation on the left position. 
 * @example deleteDataTables("Molecules", "Clusters");
 */
function deleteDataTables(datatables){
    checkWrapper("deleteDataTables", function () {
        var b = externalinterface.RemoveDataTables(datatables);
    });
    return b;
}

/**
 * Removes all relations between Data Tables from the current TIBCO Spotfire Document.
 *
 * @memberOf Data Table Manipulation
 * @example removeAllDataTableRelations();
 */
function removeAllDataTableRelations()
{
    checkWrapper("removeAllDataTableRelations", function () {
        var b = externalinterface.RemoveAllDataTableRelations();
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Removes relations between Data Tables from the current TIBCO Spotfire Document.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatables The list of data tables names implicated in relations to remove, separated by a comma.
 * @example removeDataTableRelations("Molecules");
 */
function removeDataTableRelations(datatables)
{
    checkWrapper("removeDataTableRelations", function () {
        var b = externalinterface.RemoveDataTableRelations(datatables);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Renames a Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} Name of the Data Table before renaming.
 * @param {string} Name of the Data Table after renaming.
 * @exception If the data table to rename does not exist in the document.
 * @example renameDataTable("myDataTable", "myRenamedDataTable");
 */
function renameDataTable(oldDTname, newDTname)
{
    checkWrapper("renameDataTable", function () {
        var b = externalinterface.RenameDataTable(oldDTname, newDTname);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/*************EVENT*METHODS*****************/

/**
 *  Create an event listener on the browser visible state.
 *  The callback will get two boolean parameter :
 *    - pVisible : The visible state of the browser.
 *    - pNewDocument : If the document is a new one since the last call of the callback.
 *  The button showing the Web Panel when clicked will internally set the visible state to false before true.
 *  Beware, due to the loading state of the browser at the moment of the callback avoid using alert in the callback.
 *
 *  As the browser persist on document close this event is not unregistered when calling unregisterEventsHandler.
 *  Only one callback can be registered on this event at a time.
 *  
 * @memberOf Event Management
 * @param {function} callback Javascript function called when a new document is open.
 */
function registerBrowserVisible(callback) {
    checkWrapper("registerDocumentOpen", function () {
        if( typeof(callback) !== "function" ) {
            throw new Error("First parameter is expected to be a function");
        }
        var b = externalinterface.RegisterBrowserVisible(function(pVisible, pNewDocument) {
            callback(eval(pVisible), eval(pNewDocument));
        });
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 *  Create an event listener for managing data table collection.
 *  The callback will get two parameter :
 *    - The first one defining the event occuring would be a string in :
 *        - "add" : When a new data table is added to the document.
 *        - "rename" : When a data table is renamed.
 *        - "remove" : When a data table is removed from the document.
 *    - The second parameter will be an array containing the list of the data table currently in the document.
 *
 *  As for every event registering method it is best to call unregisterEventsHandler on unload or at the start of the next page to avoid call to discarded javascript function.
 *
 * @memberOf Event Management
 * @param {function} callback Javascript function called when a modification happens on the document.
 */
function registerDataTableChange(callback) {
    checkWrapper("registerDataTableChange", function () {
        if( typeof(callback) !== "function" ) {
            throw new Error("First parameter is expected to be a function");
        }
        var b = externalinterface.RegisterDataTableChange(function(pEvent, pData) {
            callback(pEvent, eval(pData));
        });
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 *  Create an event listener for managing data column collection of a given data table.
 *  The callback will get three parameter :
 *    - The first one defining the event occuring would be a string in :
 *        - "add" : When a new data table is added to the document.
 *        - "rename" : When a data table is renamed.
 *        - "remove" : When a data table is removed from the document.
 *    - The second parameter will be the current name of the data table.
 *    - The third parameter will be an array containing the list of the column in the data table.
 *
 *  As for every event registering method it is best to call unregisterEventsHandler on unload or at the start of the next page to avoid call to discarded javascript function.
 *
 * @memberOf Event Management
 * @param {function} callback Javascript function called when a modification happens on the document.
 * @param {string} dataTableName Data table name to listen.
 */
function registerDataColumnChange(callback, dataTableName) {
    checkWrapper("registerDataColumnChange", function () {
        if( typeof(callback) !== "function" ) {
            throw new Error("First parameter is expected to be a function");
        }
        var b = externalinterface.RegisterDataColumnChange(function(pEvent, pDataTableName, pData) {
            callback(pEvent, pDataTableName, eval(pData));
        }, dataTableName);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 *  Creates an event listener for managing markings and highlighting.
 *
 *  As for every event registering method it is best to call unregisterEventsHandler on unload or at the start of the next page to avoid call to discarded javascript function.
 *
 * @memberOf Event Management
 * @param {string} eventshandler Instance ID for the eventshandler object.
 * @param {string} datatable The datatable name to handle. Can be null.
 * @param {string} marking The marking name to handle. Can be null.
 * @param {boolean} getdata Get data when the registered listener is handled.
 */
function registerEventsHandler2(eventshandler, datatable, marking, getdata) {
    checkWrapper("registerEventsHandler2", function () {
        var dummy;
        if (typeof getdata !== "undefined") {
            dummy = externalinterface.RegisterCallback(eventshandler, datatable, marking, "true", getdata);
        }
        else {
            dummy = externalinterface.RegisterCallback(eventshandler, datatable, marking, "true", true);
        }
        if(!dummy) {
            throw new Error(getError());
        }
    });
}

/**
 *  Events handler object.
 *
 * @memberOf Event Management
 */
function eventshandler(){
    try{
        this.OnMarkedRecordsChanged = onMarkRecords; //onMarkRecords : when marking changes
        this.OnHighlightChanged     = onHiLiteMarker; //onHiLiteMarker : when records are highlighted
        this.OnUnHighlightChanged   = onUnHiLiteMarker;
        this.OnStatusChanged        = onStatusChanged;
    }
    catch(error){
        if(showAlerts){
            alert("Error eventshandler: " + error.message);
        }
        else{
            throw new Error("Error eventshandler: " + error.message);
        }
    }
}

/**
 * Removes specific event listener.
 *
 * @memberOf Event Management
 * @param {string} datatable The datatable name to unregister the event from. Can be null.
 * @param {string} marking The marking name to unregister the event from. Can be null.
 */
function unregisterEventHandler(datatable, marking) {
    checkWrapper("unregisterEventHandler", function () {
        var dummy = externalinterface.UnregisterCallback(datatable, marking);
        if(!dummy) {
            throw new Error(getError());
        }
    });
}

/**
 * Removes listener on the data table (only work on column listener at the moment).
 * 
 * @memberOf Event Management
 * @param {string} datatable The datatable name to unregister the event from. Cannot be null.
 */
function unregisterEventHandler(datatable) {
    checkWrapper("unregisterEventHandler", function () {
        var success = externalinterface.UnregisterCallback(datatable);
        if(!success) {
            throw new Error(getError());
        }
    });
}

/**
 *  Kills event listener process.
 *
 * @memberOf Event Management
 */
function unregisterEventsHandler() {
    checkWrapper("unregisterEventsHandler", function () {
        var dummy = externalinterface.UnregisterCallbacks();
        if(!dummy) {
            throw new Error(getError());
        }
    });
}

/*************VISUALIZATION*METHODS******************************************************************************************************/

/**
 *  Clears all visuals of a page of the current TIBCO Spotfire document.
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} pagetitle The title of TIBCO Spotfire Page containing visuals to clear. If empty, the Spotfire active Page is used.
 */
function clearVisuals(pagetitle){
    checkWrapper("clearVisuals", function () {
        var b = externalinterface.ClearVisuals(pagetitle);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Table Plot visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createtable('1', 'MyTable', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createtable(id, title, datatable, page){
    checkWrapper("createtable", function () {
        var b = externalinterface.CreateTable(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Bar Chart visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createbarchart('1', 'MyBarChart', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createbarchart(id, title, datatable, page){
    checkWrapper("createbarchart", function () {
        var b = externalinterface.CreateBarchart(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Scatter Plot visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createscatterplot('1', 'MyScatterPlot', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createscatterplot(id, title, datatable, page){
    checkWrapper("createscatterplot", function () {
        var b = externalinterface.CreateScatterplot(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Line Chart visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createlinechart('1', 'MyLineChart', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createlinechart(id, title, datatable, page){
    checkWrapper("createlinechart", function () {
        var b = externalinterface.CreateLinechart(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Combination Chart visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createcombinationchart('1', 'MyCombinationChart', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createcombinationchart(id, title, datatable, page){
    checkWrapper("createcombinationchart", function () {
        var b = externalinterface.CreateCombinationchart(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Pie Chart visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createpiechart('1', 'MyPieChart', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createpiechart(id, title, datatable, page){
    checkWrapper("createpiechart", function () {
        var b = externalinterface.CreatePiechart(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Scatter Plot 3D visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createscatterplot3d('1', 'My3DScatterPlot', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createscatterplot3d(id, title, datatable, page){
    checkWrapper("createscatterplot3d", function () {
        var b = externalinterface.CreateScatterplot3D(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Treemap visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createtreemap('1', 'MyTreemap', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createtreemap(id, title, datatable, page){
    checkWrapper("createtreemap", function () {
        var b = externalinterface.CreateTreemap(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Heat Map visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createheatmap('1', 'MyHeatMap', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createheatmap(id, title, datatable, page){
    checkWrapper("createheatmap", function () {
        var b = externalinterface.CreateHeatmap(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Cross Table visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createcrosstable('1', 'MyCrossTable', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createcrosstable(id, title, datatable, page){
    checkWrapper("createcrosstable", function () {
        var b = externalinterface.CreateCrosstable(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Creates a Box Plot visualization (but not display it. Use showvisual method to display all created visualizations).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title. If empty, the Spotfire active Page is used.
 * @param {string} datatable The name of the TIBCO Spotfire Data Table. If empty, the Spotfire active data table is used.
 * @param {string} page The name of Tiboc Spotfire Page.
 * @exception If the id is null or empty.
 * @example createboxplot('1', 'MyBoxPlot', 'Pipeline Pilot Data 1', 'Discngine Web Panel Start Page');
 */
function createboxplot(id, title, datatable, page){
    checkWrapper("createboxplot", function () {
        var b = externalinterface.CreateBoxPlot(id, title, datatable, page);
        if(!b) {
            throw new Error(getError());
        }
    });
}

/**
 * Sets general properties of a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - General
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} title The Table Plot visualization title.
 * @param {boolean} showtitlebar Show visualization title bar.
 * @param {string} description The TIBCO Spotfire visualization description.
 * @example setgeneralproperties('1', 'Scatter Plot', 'True', 'This is my scatter plot');
 */
function setgeneralproperties(id, title, showtitlebar, description){
    checkWrapper("setgeneralproperties", function () {
        var b = externalinterface.SetGeneralProperties(id, title, showtitlebar, description);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the marking limiting data shown in the visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Data
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} markerby The name of the Marking to use.
 * @example setmarkerbyproperties('1', 'Marking');
 */
function setmarkerbyproperties(id, markerby){
    checkWrapper("setmarkerbyproperties", function () {
        var b = externalinterface.SetmarkerbyProperties(id, markerby);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets Appearance properties for a Bar Chart visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} stackmode The Layout property : 0=Side-by-Side bars, 1=Stacked bars, 2=100% stacked bars (use null for default value).
 * @param {integer} barswidth The Bar width property, msut be between 0 and 100 (use null for default value).
 * @param {boolean} showshadowsbars Show shadows indicating the unfiltered data (use null for default value).
 * @param {boolean} sortbars Sort bars by value (use null for default value).
 */
function setlayoutproperties(id, stackmode, barswidth, showshadowsbars, sortbars){
    checkWrapper("setlayoutproperties", function () {
        var b = externalinterface.SetLayoutProperties(id, stackmode, barswidth, showshadowsbars, sortbars);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets Jittering properties for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {double} xjitter The Jitter property for X axis, must be between 0 and 0.5 (use null for default value).
 * @param {double} yjitter The Jitter property for Y axis, must be between 0 and 0.5 (use null for default value).
 * @param {double} zjitter The Jitter property for Z axis, must be between 0 and 0.5 (use null for default value).
 * @param {boolean} samejitter Use the same amount of jittering for all axises (use null for default value).
 * @example setjitterproperties('1', 0, 0, 0, false);
 */
function setjitterproperties(id, xjitter, yjitter, zjitter, samejitter){
    checkWrapper("setjitterproperties", function () {
        var b = externalinterface.SetJitterProperties(id, xjitter, yjitter, zjitter, samejitter);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sorts sectors by size property for a Pie Chart visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} sortsectorbysize Sort sectors by size property.
 */
function sortsectorsbysize(id, sortsectorbysize){
    checkWrapper("sortsectorsbysize", function () {
        var b = externalinterface.SortSectorsbySize(id, sortsectorbysize);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Shows navigation controls property for a Scatter Plot 3D Chart visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} shownavigationcontrols Shows navigation controls property.
 */
function shownavigationcontrols(id, shownavigationcontrols){
    checkWrapper("shownavigationcontrols", function () {
        var b = externalinterface.ShowNavigationControls(id, shownavigationcontrols);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Shows gridlines Category Axis property for a Bar Chart visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} showgridlines Shows gridlines property.
 */
function showgridlines(id, showgridlines){
    checkWrapper("showgridlines", function () {
        var b = externalinterface.ShowGridlines(id, showgridlines);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets grand total for rows property for a Cross Table visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} grandtotalforrows Grand total for rows property.
 */
function grandtotalforrows(id, grandtotalforrows){
    checkWrapper("grandtotalforrows", function () {
        var b = externalinterface.GrandTotalforRows(id, grandtotalforrows);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets grand total for columns property for a Cross Table visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} grandtotalforrows Grand total for columns property.
 */
function grandtotalforcolumns(id, grandtotalforcolumns){
    checkWrapper("grandtotalforcolumns", function () {
        var b = externalinterface.GrandTotalForColumns(id, grandtotalforcolumns);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Allows table data export property for a Cross Table and Table Plot visualizations.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} allowtabledataexport Allow table data export.
 */
function allowtabledataexport(id, allowtabledataexport){
    checkWrapper("allowtabledataexport", function () {
        var b = externalinterface.AllowTableDataExport(id, allowtabledataexport);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets header row height (number of lines) property for a Table Plot visualizations.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} headerrowheight Header row height.
 */
function setheaderrowheight(id, headerrowheight){
    checkWrapper("setheaderrowheight", function () {
        var b = externalinterface.SetHeaderRowHeight(id, headerrowheight);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets data rows height (number of lines) property for a Table Plot visualizations.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} datarowsheight Data rows height.
 */
function setdatarowsheight(id, datarowsheight){
    checkWrapper("setdatarowsheight", function () {
        var b = externalinterface.SetDataRowsHeight(id, datarowsheight);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets number of frozen columns property for a Table Plot visualizations.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} frozencolumnscount Number of frozen columns.
 */
function setfrozencolumnscount(id, frozencolumnscount){
    checkWrapper("setfrozencolumnscount", function () {
        var b = externalinterface.SetFrozenColumnsCount(id, frozencolumnscount);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets columns to display property for Table Plot visualizations.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Columns
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} displaycolumnsmode The chosen mode to display columns : "All" or "Custom".
 * @param {string} visiblecolumns List of columns to display (in case of "Custom" mode) with a comma as separator.
 */
function setselectedcolumns(id, displaycolumnsmode, selectedcolumns) {
    checkWrapper("setselectedcolumns", function () {
        var b = externalinterface.SetSelectedColumns(id, displaycolumnsmode, selectedcolumns);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the Auto Add New Columns property of a TIBCO Spotfire Table Plot.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Columns
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} autoadd Automatically add new columns ?
 */
function setAutoAddNewColumns(id, autoadd) {
    checkWrapper("setAutoAddNewColumns", function () {
        var dummy = externalinterface.SetAutoAddNewColumns(id, autoadd);
        if(!dummy) { throw new Error(getError()); }
    });
}

/**
 * @deprecated
 * Sets Appearance properties for a Line Chart visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} linewidth The Line width property, must be between 0 and 10 (use null for default value).
 * @param {booolean} showmarkers Show marker property (use null for default value).
 * @param {boolean} markersize Marker size, must be between 0 and 100 (use null for default value).
 * @param {boolean} breakonempty Break lines on empty (use null for default value).
 */
function setlineproperties(id, linewidth, showmarkers, markersize, breakonempty){
    checkWrapper("setlineproperties", function () {
        var b = externalinterface.SetLineproperties(id, linewidth, showmarkers, markersize, breakonempty);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets box width property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} boxwidth The Line width property, must be between 0 and 10.
 */
function setboxwith(id, boxwidth){
    checkWrapper("setboxwith", function () {
        var b = externalinterface.SetBoxWith(id, boxwidth);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets jittering property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {double} value Jittering property, must be between 0 and 0.5.
 */
function setjitteroutervalue(id, value){
    checkWrapper("setjitteroutervalue", function () {
        var b = externalinterface.SetJitterOuterValue(id, value);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Shows distribution property for a Box Plot visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} value Show distribution property.
 */
function showdistribution(id, value){
    checkWrapper("showdistribution", function () {
        var b = externalinterface.ShowDistribution(id, value);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Shows 95% confidence interval of the mean for a Box Plot visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} value Show 95% confidence interval.
 */
function show95CI(id, value){
    checkWrapper("show95CI", function () {
        var b = externalinterface.Show95PercentCI(id, value);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Shows comparison circles for a Box Plot visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} show Shows comparison circles.
 * @param {double} value Alpha level value.
 */
function showcomparisoncircles(id, show, value){
    checkWrapper("showcomparisoncircles", function () {
        var b = externalinterface.ShowComparisonCircles(id, show, value);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Uses relative scale for a Box Plot visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Appearance
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} value Relative scale.
 */
function showrelativescale(id, value){
    checkWrapper("showrelativescale", function () {
        var b = externalinterface.UseRelativeScale(id, value);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets axis properties for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis (X, Y or Z).
 * @param {string} expression Axis expression (with aggregation method if needed).
 * @param {integer} bins Number of bins (use "null" if not needed).
 * @example setaxis('1', 'X', 'ALogP', 'null');
 * @example setaxis('1', 'Y', 'ALogP', '20');
 */
function setaxis(id, axis, expression, bins){
    checkWrapper("setaxis", function () {
        var b = externalinterface.SetAxis(id, axis, expression, bins);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets axis zoom range for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis.
 * @param {string} zoomrangelow Min zoom range.
 * @param {string} zoomrangehigh Max zoom range.
 */
function setzoomrangeaxis(id, axis, zoomrangelow, zoomrangehigh){
    checkWrapper("setzoomrangeaxis", function () {
        var b = externalinterface.SetZoomRangeAxis(id, axis, zoomrangelow, zoomrangehigh);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets axis include zero in auto zoom for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis (X, Y or Z).
 * @param {boolean} includezeroinautozoom Include origin.
 * @exception Message error, use getError method to get the error message.
 */
function setaxisincludezeroinautozoom(id, axis, includezeroinautozoom){
    checkWrapper("setaxisincludezeroinautozoom", function () {
        var b = externalinterface.SetAxisIncludezeroInAutozoom(id, axis, includezeroinautozoom);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Shows axis manuel zoom property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis.
 * @param {boolean} manuelzoom Show zoom slider.
 * @example setmanuelzoomaxis('1', 'X', 'False');
 */
function setmanuelzoomaxis(id, axis, manuelzoom){
    checkWrapper("setmanuelzoomaxis", function () {
        var b = externalinterface.SetManuelzoomAxis(id, axis, manuelzoom);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Shows axis gridlines property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis.
 * @param {boolean} showgridlines Show gridlines.
 * @example setshowgridllinesaxis('1', 'X', 'True');
 */
function setshowgridllinesaxis(id, axis, showgridlines){
    checkWrapper("setshowgridllinesaxis", function () {
        var b = externalinterface.SetShowGridllinesAxis(id, axis, showgridlines);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Apply a Log 10 transform to an axis.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis ('X', 'Y', or 'Z').
 * @param {boolean} logscale Apply a Log 10 transform?
 * @example setLogScaleAxis('1', 'X', 'True');
 */
function setLogScaleAxis(id, axis, logscale)
{
    checkWrapper("setLogScaleAxis", function () {
        var b = externalinterface.SetLogScaleAxis(id, axis, logscale);
        if(!b) { throw new Error(getError()); }
    });
}


/**
 * Set axis reverse scale property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis ('X', 'Y', or 'Z').
 * @param {boolean} reversed Reverse scale.
 * @example setreversedaxis('1', 'X', 'True');
 */
function setreversedaxis(id, axis, reversed){
    checkWrapper("setreversedaxis", function () {
        var b = externalinterface.SetReversedAxis(id, axis, reversed);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Shows axis labels property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis.
 * @param {boolean} showlabels Show labels.
 * @example setshowlabelsaxis('1', 'X', 'True');
 */
function setshowlabelsaxis(id, axis, showlabels){
    checkWrapper("setshowlabelsaxis", function () {
        var b = externalinterface.SetShowLabelsAxis(id, axis, showlabels);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets axis labels orientation property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis.
 * @param {integer} labelorientation Labels orientaiton: 0=horizontal, 1=vertical.
 * @example setlabelorientationaxis('1', 'X', 1);
 */
function setlabelorientationaxis(id, axis, labelorientation){
    checkWrapper("setlabelorientationaxis", function () {
        var b = externalinterface.SetLabelOrientationAxis(id, axis, labelorientation);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets axis Max number of labels for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis.
 * @param {integer} maximumnumberofticks Max number of labels.
 * @example setmaximumnumberofticksaxis('1', 'X', 15);
 */
function setmaximumnumberofticksaxis(id, axis, maximumnumberofticks){
    checkWrapper("setmaximumnumberofticksaxis", function () {
        var b = externalinterface.SetMaximumNumberOfTicksAxis(id, axis, maximumnumberofticks);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets axis position for a Heat Map visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Axis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis.
 * @param {integer} axisposition Axis position: 0=top or right, 1=bottom or left.
 * @example setaxisposition("1", "X", 0);
 */
function setaxisposition(id, axis, axisposition){
    checkWrapper("setaxisposition", function () {
        var b = externalinterface.SetAxisPosition(id, axis, axisposition);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets axis error bars for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Error Bars
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis.
 * @param {string} errorbars_lowerexpression Lower error expression (with aggregation method if needed).
 * @param {string} errorbars_uppererexpression Upper error expression (with aggregation method if needed).
 */
function seterrorbarsaxis(id, axis, errorbars_lowerexpression,errorbars_uppererexpression){
    checkWrapper("seterrorbarsaxis", function () {
        var b = externalinterface.SetErrorbarsAxis(id, axis, errorbars_lowerexpression, errorbars_uppererexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets axis scale labels font for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Fonts
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} axis Axis.
 * @param {string} labelsfontname Labels font name: Arial, Arial Black, Comic Sans MS, Courier New...
 * @param {integer} labelsfontsize Labels font size.
 * @param {integer} labelsfontstyle Labels font style: 0=Regular, 1=Italic, 2=Bold, 3= Bold Italic
 */
function setlabelsfontaxis(id, axis, labelsfontname, labelsfontsize, labelsfontstyle){
    checkWrapper("setlabelsfontaxis", function () {
        var b = externalinterface.SetLabelsFontAxis(id, axis, labelsfontname, labelsfontsize, labelsfontstyle);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets labels font for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Fonts
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} labelsfontname Labels font name: Arial, Arial Black, Comic Sans MS, Courier New...
 * @param {integer} labelsfontsize Labels font size.
 * @param {integer} labelsfontstyle Labels font style: 0=Regular, 1=Italic, 2=Bold, 3= Bold Italic
 */
function setlabelsfont(id, labelsfontname, labelsfontsize, labelsfontstyle){
    checkWrapper("setlabelsfont", function () {
        var b = externalinterface.SetLabelsFont(id, labelsfontname, labelsfontsize, labelsfontstyle);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets legend font for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Fonts
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} trellisheadersfontname Legend font name: Arial, Arial Black, Comic Sans MS, Courier New...
 * @param {integer} trellisheadersfontsize Legend font size.
 * @param {integer} trellisheadersfontstyle Legend font style: 0=Regular, 1=Italic, 2=Bold, 3= Bold Italic
 */
function setlegendfont(id, legendfontname, legendfontsize, legendfontstyle){
    checkWrapper("setlegendfont", function () {
        var b = externalinterface.SetLegendFont(id, legendfontname, legendfontsize, legendfontstyle);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets line curve labels font for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Fonts
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} linecurvelabelsfontname line curve labels font name: Arial, Arial Black, Comic Sans MS, Courier New...
 * @param {integer} linecurvelabelsfontsize line curve labels font size.
 * @param {integer} linecurvelabelsfontstyle line curve labels font style: 0=Regular, 1=Italic, 2=Bold, 3= Bold Italic
 */
function setlinecurvelabelsfont(id, linecurvelabelsfontname, linecurvelabelsfontsize, linecurvelabelsfontstyle){
    checkWrapper("setlinecurvelabelsfont", function () {
        var b = externalinterface.SetLineCurveLabelsFont(id, linecurvelabelsfontname, linecurvelabelsfontsize, linecurvelabelsfontstyle);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets trellis headers font for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Trellis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} trellisheadersfontname Trellis header font name: Arial, Arial Black, Comic Sans MS, Courier New...
 * @param {integer} trellisheadersfontsize Trellis header font size.
 * @param {integer} trellisheadersfontstyle Trellis header font style: 0=Regular, 1=Italic, 2=Bold, 3= Bold Italic
 */
function settrellisheadersfont(id, trellisheadersfontname, trellisheadersfontsize, trellisheadersfontstyle){
    checkWrapper("settrellisheadersfont", function () {
        var b = externalinterface.SetTrellisHeadersFont(id, trellisheadersfontname, trellisheadersfontsize, trellisheadersfontstyle);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets line by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Line By
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} linebyexpression Line by expression (with aggregation method if needed).
 */
function setlineby(id, linebyexpression){
    checkWrapper("setlineby", function () {
        var b = externalinterface.SetLineBy(id, linebyexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets default color property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Colors
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} defaultcolor Default color name.
 * @example setdefaultcolor('3', 'CornflowerBlue');
 */
function setdefaultcolor(id, defaultcolor){
    checkWrapper("setdefaultcolor", function () {
        var b = externalinterface.SetDefaultColor(id, defaultcolor);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets color by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Colors
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} urlfile XML file containing the dxp ColorScheme.
 * @example setcolorfromfile('3', 'path/ColorScheme.dxpcolor');
 */
function setcolorfromfile(id, urlfile){
    checkWrapper("setcolorfromfile", function () {
        var b = externalinterface.SetColorFromFile(id, urlfile);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets color by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Colors
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} librarypath XML file containing the dxp ColorScheme in the library.
 */
function setcolorfromlibrary(id, librarypath){
    checkWrapper("setcolorfromlibrary", function () {
        var b = externalinterface.SetColorFromLibrary(id, librarypath);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets color by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Colors
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} colorbyexpression Color by expression (with aggregation method if needed).
 */
function setcolorby(id, colorbyexpression){
    checkWrapper("setcolorby", function () {
        var b = externalinterface.SetColorBy(id, colorbyexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a color rule for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Colors
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} idpoint The id of the color rule.
 * @param {string} type the rule type for the color rule. Available values depend of the TIBCO Spotfire Color Axis Mode.
 * @param {string} value The value for the color rule.
 * @param {string} color The color associated with the value of the color rule.
 */
function addcolorrule(id, idrule, type, value, color){
    checkWrapper("addcolorrule", function () {
        var b = externalinterface.AddColorRule(id, idrule, type, value, color);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a breakpoint color rule for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Colors
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} idpoint The id of the breakpoint color rule.
 * @param {string} type the rule type for the breakpoint color rule. Can be "Value", "Min", "Max", "Average", "Median", "Percent", "Percentile", "Custom expression".
 * @param {string} value The value for the breakpoint color rule (null if the rule type is "Min", "Max", "Average" or "Median").
 * @param {string} color The color associated with the value of the breakpoint color rule.
 */
function addbreakpointcolorrule(id, idpoint, type, value, color){
    checkWrapper("addbreakpointcolorrule", function () {
        var b = externalinterface.AddBreakpointColorRule(id, idpoint, type, value, color);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets axis mode property (from color by propoerty) for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Colors
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} coloraxismode Color axis mode, "True" for Categorical, "False" for Continuous.
 */
function setcoloraxismode(id, coloraxismode){
    checkWrapper("setcoloraxismode", function () {
        var b = externalinterface.SetColorAxisMode(id, coloraxismode);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets color mode property (from color by propoerty) for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Colors
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} colorcolormode Color color mode, it can only be "Categorical", "Fixed, "Gradient", "Segments", "Unique values".
 */
function setcolorcolormode (id, colorcolormode){
    checkWrapper("setcolorcolormode", function () {
        var b = externalinterface.SetColorColorMode(id, colorcolormode);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets marker size property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Size
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {Double} markersize Marker size, must be between 0 and 100.
 */
function setmarkersize(id, markersize){
    checkWrapper("setmarkersize", function () {
        var b = externalinterface.SetMarkerSize(id, markersize);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets size by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Size
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} sizebyexpression Size by expression (with aggregation method if needed).
 */
function setsizeby(id, sizebyexpression){
    checkWrapper("setsizeby", function () {
        var b = externalinterface.SetSizeBy(id, sizebyexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets size scale property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Size
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} scalemethod Size scale method: false=From min to max, true=From 0 to max.
 * @param {boolean} maxlimitscalemethod Max limit method: false=Value, true=Automatic (use null for default value).
 * @param {string} maxlimitscalevalue Max limit value (use null for default value).
 * @param {boolean} minlimitscalemethod Min limit method: false=Value, true=Automatic (use null for default value).
 * @param {string} minlimitscalevalue Min limit value (use null for default value).
 */
function setsizescale(id, scalemethod, maxlimitscalemethod, maxlimitscalevalue, minlimitscalemethod, minlimitscalevalue){
    checkWrapper("setsizescale", function () {
        var b = externalinterface.SetSizeScale(id, scalemethod, maxlimitscalemethod, maxlimitscalevalue, minlimitscalemethod, minlimitscalevalue);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets sector size by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Size
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} sectorsizebyexpression Sector size by expression (with aggregation method if needed).
 */
function setsectorsizeby(id, sectorsizebyexpression){
    checkWrapper("setsectorsizeby", function () {
        var b = externalinterface.SetSectorSizeBy(id, sectorsizebyexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets hierarchy by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Hierarchy
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} hierarchybyexpression Hierarchy by expression (with aggregation method if needed).
 * @param {integer} bins Bins property.
 */
function sethierarchy(id, hierarchybyexpression, bins){
    checkWrapper("sethierarchy", function () {
        var b = externalinterface.SetHierarchy(id, hierarchybyexpression, bins);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets fixed shape property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} defaultshape Fixed shape: 0=Circle, 1=Diamond, 2=Pentagon, 3=Square, 4=StarFive, 5= StarFour, 6=TriangleDown, 7=TriangleLeft, 8=TriangleRight, 9=TriangleUp
 */
function setfixedshape(id, defaultshape){
checkWrapper("setfixedshape", function () {
        var b = externalinterface.SetFixedShape(id, defaultshape);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets shape by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} shapebyexpression Shape by expression (with aggregation method if needed).
 */
function setshapeby(id, shapebyexpression){
    checkWrapper("setshapeby", function () {
        var b = externalinterface.SetShapeBy(id, shapebyexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets sector size by property for a visualization (only for the "Pies" method).
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} shapesectorsizeby Sector size by expression (an aggregation method is mandatory).
 */
function setShapeSectorSizeBy(id, shapesectorsizeby){
    checkWrapper("setShapeSectorSizeBy", function () {
        var b = externalinterface.SetShapeSectorSizeBy(id, shapesectorsizeby);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the sector value property for a visualization (only for the "Pies" method).
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} shapesectorvalue True to show the sector value in labels, false otherwise.
 */
function setShapeSectorValue(id, shapesectorvalue){
    checkWrapper("setShapeSectorValue", function () {
        var b = externalinterface.SetShapeSectorValue(id, shapesectorvalue);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the sector category property for a visualization (only for the "Pies" method).
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} shapesectorcategory True to show the sector category in labels, false otherwise.
 */
function setShapeSectorCategory(id, shapesectorcategory){
    checkWrapper("setShapeSectorCategory", function () {
        var b = externalinterface.SetShapeSectorCategory(id, shapesectorcategory);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the sector percentage property for a visualization (only for the "Pies" method).
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} shapesectorpercentage True to show the sector percentage in labels, false otherwise.
 */
function setShapeSectorPercentage(id, shapesectorpercentage){
    checkWrapper("setShapeSectorPercentage", function () {
        var b = externalinterface.SetShapeSectorPercentage(id, shapesectorpercentage);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the sector percentage threshold property for a visualization (only for the "Pies" method).
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {int} shapesectorpercentagethreshold Value indicating the limit for label percentage. Labels with a percentage above this value will be rendered.
 */
function setShapeSectorPercentageThreshold(id, shapesectorpercentagethreshold){
    checkWrapper("setShapeSectorPercentageThreshold", function () {
        var b = externalinterface.SetShapeSectorPercentageThreshold(id, shapesectorpercentagethreshold);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the sector percentage decimals property for a visualization (only for the "Pies" method).
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {int} shapesectorpercentagedecimals Number of decimal digits for percentages.
 */
function setShapeSectorPercentageDecimals(id, shapesectorpercentagedecimals){
    checkWrapper("setShapeSectorPercentageDecimals", function () {
        var b = externalinterface.SetShapeSectorPercentageDecimals(id, shapesectorpercentagedecimals);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the label position property for a visualization (only for the "Pies" method).
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {int} shapelabelposition The label position: 0 = Inside, 1 = Outside.
 */
function setShapeLabelPosition(id, shapelabelposition){
    checkWrapper("setShapeLabelPosition", function () {
        var b = externalinterface.SetShapeLabelPosition(id, shapelabelposition);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the sort sectors by size property for a visualization (only for the "Pies" method).
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Shape
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} shapesortsectorsbysize True to sort sectors by size, false otherwise.
 */
function setShapeSortSectorsBySize(id, shapesortsectorsbysize){
    checkWrapper("setShapeSortSectorsBySize", function () {
        var b = externalinterface.SetShapeSortSectorsBySize(id, shapesortsectorsbysize);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets series by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Series
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} seriesbyexpression Series by expression (with aggregation method if needed).
 * @param {boolean} seriestype Series type: true=Lines, false=Bars.
 */
function setseriesby(id, seriesbyexpression, seriestype){
    checkWrapper("setseriesby", function () {
        var b = externalinterface.SetSeriesBy(id, seriesbyexpression, seriestype);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets label properties for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} labelvisibility Label visibility: 0=All, 1=Marked rows, 2=None (use null for default value).
 * @param {boolean} labelorientation Label orientation: 0=Horizontal, 1=Vertical (use null for default value).
 * @param {boolean} labelcompletebar Label complete Bar (use null for default value).
 * @param {boolean} labelsegment Label segment (use null for default value).
 * @param {integer} segmentlabelinformationtype Segment label information type: 0=Percentage, 1=Value (use null for default value).
 * @param {integer} maxnumberoflabels Max number of labels, must be between 0 and 200 (use null for default value).
 * @param {integer} labelpercentagedecimaldigits Percentage decimals digits, must be between 0 and 6 (use null for default value).
 */
function setlabelproperties(id, labelvisibility, labelorientation, labelcompletebar, labelsegment,
                            segmentlabelinformationtype, maxnumberoflabels, labelpercentagedecimaldigits){
   checkWrapper("setlabelproperties", function () {
        var b = externalinterface.SetLabelProperties(id, labelvisibility, labelorientation, labelcompletebar, labelsegment,
                                                         segmentlabelinformationtype, maxnumberoflabels, labelpercentagedecimaldigits);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets show in labels property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} showlinelabels Show line labels (use null for default value).
 * @param {boolean} showindividualvalues Show individual values (use null for default value).
 */
function showinlabels(id, showlinelabels, showindividualvalues){
   checkWrapper("showinlabels", function () {
        var b = externalinterface.ShowInLabels(id, showlinelabels, showindividualvalues);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets show labels for property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} labelvisibility Labels visibility: 0=All, 1=Marked rows, 2=None.
 */
function showlabelsfor(id, labelvisibility){
    checkWrapper("showlabelsfor", function () {
        var b = externalinterface.ShowlLabelsFor(id, labelvisibility);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets max number of labels for property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} maxnumberoflabels Max number of labels, must be between 0 to 200.
 */
function setmaxnumberoflabels(id, maxnumberoflabels){
    checkWrapper("setmaxnumberoflabels", function () {
        var b = externalinterface.SetMaxnumberOfLabels(id, maxnumberoflabels);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets label by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} labelbyexpression Label by expression (with aggregation method if needed).
 */
function setlabelby(id, labelbyexpression){
    checkWrapper("setlabelby", function () {
        var b = externalinterface.SetLabelBy(id, labelbyexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets show bar labels property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} showbarlabels Show bar labels.
 */
function showbarlabels(id, showbarlabels){
    checkWrapper("showbarlabels", function () {
        var b = externalinterface.ShowBarLabels(id, showbarlabels);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets label orientation property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} labelorientation Labels orientation: 0=Horizontal, 1=Vertical.
 */
function setlabelorientation(id, labelorientation){
    checkWrapper("setlabelorientation", function () {
        var b = externalinterface.SetLabelOrientation(id, labelorientation);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets show line marker labels property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} showlinemarkerlabels Show line marker labels.
 */
function showlinemarkerlabels(id, showlinemarkerlabels){
    checkWrapper("showlinemarkerlabels", function () {
        var b = externalinterface.ShowLineMarkerLabels(id, showlinemarkerlabels);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets show sector in labels property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} sectorpercentage Sector percentage (use null for default value).
 * @param {boolean} sectorvalue Sector value (use null for default value).
 * @param {boolean} sectorcategory Sector category (use null for default value).
 */
function showsectorinlabels(id, sectorpercentage, sectorvalue, sectorcategory){
    checkWrapper("showsectorinlabels", function () {
        var b = externalinterface.ShowSectorInLabels(id, sectorpercentage, sectorvalue, sectorcategory);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets sector percentage threshold property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} threshold Sector percentage threshold.
 */
function setsectorpercentagethreshold(id, threshold){
    checkWrapper("setsectorpercentagethreshold", function () {
        var b = externalinterface.SetSectorPercentageThreshold(id, threshold);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets percentage decimals property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} decimals Percentage decimals.
 */
function setpercentagedecimals(id, decimals){
    checkWrapper("setpercentagedecimals", function () {
        var b = externalinterface.SetPercentageDecimals(id, decimals);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets hierarchy headers property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} showhierarchyheaders Show hierarchy headers.
 */
function showhierarchyheaders(id, showhierarchyheaders){
    checkWrapper("showhierarchyheaders", function () {
        var b = externalinterface.showhierarchyheaders(id, ShowHierarchyHeaders);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets show labels property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Labels
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} showlabels Show labels.
 */
function showlables(id, showlabels){
    checkWrapper("showlables", function () {
        var b = externalinterface.ShowLables(id, showlabels);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets rows, columns and pages trellis properties for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Trellis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} trelliscolumnsexpression Trellis columns expression.
 * @param {string} trellisrowsexpression Trellis rows expression.
 * @param {string} trellispagesexpression Trellis pages expression.
 */
function setrowsandcolumntrellis(id, trelliscolumnsexpression, trellisrowsexpression, trellispagesexpression){
    checkWrapper("setrowsandcolumntrellis", function () {
        var b = externalinterface.SetRowsAndColumnTrellis(id, trelliscolumnsexpression, trellisrowsexpression, trellispagesexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets panel split by trellis properties for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Trellis
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} trellispanelsexpression Trellis panel split by expression.
 * @param {boolean} trellismanuellayout Trellis manual layout (use null for default value).
 * @param {integer} trellismanuelrowcount Max number of rows (use null for default value).
 * @param {integer} trellismanuelcolumncount Max number of columns (use for null default value).
 */
function setpanelssplitbytrellis(id, trellispanelsexpression, trellismanuellayout, trellismanuelrowcount, trellismanuelcolumncount){
    checkWrapper("setpanelssplitbytrellis", function () {
        var b = externalinterface.SetPanelsSplitByTrellis(id, trellispanelsexpression, trellismanuellayout, trellismanuelrowcount, trellismanuelcolumncount);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets Line Connection Draw by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Line Connection
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} drawByExpression Column name on which draw a separate line per value.
 * Force Categorical mode by arrounding name with "<" and ">" characters.
 * @param {integer} bins Number of bins (use "null" if not needed).
 * @example setLineConnectionDrawBy("1", "CLUSTER", 5);
 * @example setLineConnectionDrawBy("1", "<CLUSTER>", "null");
 */
function setLineConnectionDrawBy(id, drawByExpression, bins){
    checkWrapper("setLineConnectionDrawBy", function () {
        var b = externalinterface.SetLineConnectionDrawBy(id, drawByExpression, bins);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets Line Connection Order By property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Line Connection
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} orderByExpression Column name on which order each line. Aggregation methods are allowed. Can be empty.
 * @example setLineConnectionOrderBy("1", "Avg(CL_SIZE)");
 */
function setLineConnectionOrderBy(id, orderByExpression){
    checkWrapper("setLineConnectionOrderBy", function () {
        var b = externalinterface.SetLineConnectionOrderBy(id, orderByExpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets Line Connection Show order arrows property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Line Connection
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} showArrows Show arrows to indicate connection order ?
 * @example setLineConnectionShowOrderArrows("1", false);
 */
function setLineConnectionShowOrderArrows(id, showArrows){
    checkWrapper("setLineConnectionShowOrderArrows", function () {
        var b = externalinterface.SetLineConnectionShowOrderArrows(id, showArrows);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets Line Connection Color property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Line Connection
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} color The color, either in the HTML format or its name, by which the Line Connection is rendered.
 * Can be empty, default is black.
 * @example setLineConnectionColor("1", "#FF0000");
 */
function setLineConnectionColor(id, color)
{
    checkWrapper("setLineConnectionColor", function () {
        var b = externalinterface.SetLineConnectionColor(id, color);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets Line Connection Use Marker Color property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Line Connection
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} useMarkerColor Use the Marker Color to color the Line Connection ?
 * @example setLineConnectionUseMarkerColor("1", false);
 */
function setLineConnectionUseMarkerColor(id, useMarkerColor)
{
    checkWrapper("setLineConnectionUseMarkerColor", function () {
        var b = externalinterface.SetLineConnectionUseMarkerColor(id, useMarkerColor);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets Line Connection Width property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Line Connection
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {float} width Width of the Line Connection, between 1 and 6.
 * @example setLineConnectionWidth("1", 1);
 */
function setLineConnectionWidth(id, width)
{
    checkWrapper("setLineConnectionWidth", function () {
        var b = externalinterface.SetLineConnectionWidth(id, width);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets Line Connection Is Background property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Line Connection
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} isBackground Place the Line Connection in background ?
 * @example setLineConnectionIsBackground("1", false);
 */
function setLineConnectionIsBackground(id, isBackground)
{
    checkWrapper("setLineConnectionIsBackground", function () {
        var b = externalinterface.SetLineConnectionIsBackground(id, isBackground);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets marker by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Marker By
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} markerbyexpression Marker by expression.
 */
function setmarkerby(id, markerbyexpression){
    checkWrapper("setmarkerby", function () {
        var b = externalinterface.SetMarkerBy(id, markerbyexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets statistics measures property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Statistics Table
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string}  value List of measures.
 */
function setstatisticsmeasures(id, value){
    checkWrapper("setstatisticsmeasures", function () {
        var b = externalinterface.SetStatisticMeasures(id, value);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets sort by property for a visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Statistics Table
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} value Sort by property.
 * @param {integer} mode Sort by mode: 0=Ascending, 1=Descending.
 */
function setsortby(id, value, mode){
    checkWrapper("setsortby", function () {
        var b = externalinterface.SetSortBy(id, value, mode);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip display mode for a visualization.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {integer} mode The mode to use to display tooltip, 0 = "Value names and values" ; 1 = "Visualization properties and values" ; can be null.
 * @example setTooltipDisplayMode('1', 0);
 */
function setTooltipDisplayMode(id, mode){
    checkWrapper("setTooltipDisplayMode", function () {
        var b = externalinterface.SetTooltipDisplayMode(id, mode);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Marker by property visible or not for a visualization.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Marker by property in tooltip ?
 * @example setTooltipMarkerByVisible('1', 'True');
 */
function setTooltipMarkerByVisible(id, visible){
    checkWrapper("setTooltipMarkerByVisible", function () {
        var b = externalinterface.SetTooltipMarkerByVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Shape by property visible or not for a visualization.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Shape by property in tooltip ?
 * @example setTooltipShapeByVisible('1', 'True');
 */
function setTooltipShapeByVisible(id, visible){
    checkWrapper("setTooltipShapeByVisible", function () {
        var b = externalinterface.SetTooltipShapeByVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Color by property visible or not for a visualization.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Color by property in tooltip ?
 * @example setTooltipColorByVisible('1', 'True');
 */
function setTooltipColorByVisible(id, visible){
    checkWrapper("setTooltipColorByVisible", function () {
        var b = externalinterface.SetTooltipColorByVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Size by property visible or not for a visualization.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Size by property in tooltip ?
 * @example setTooltipSizeByVisible('1', 'True');
 */
function setTooltipSizeByVisible(id, visible){
    checkWrapper("setTooltipSizeByVisible", function () {
        var b = externalinterface.SetTooltipSizeByVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip X property visible or not for a visualization.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show X property in tooltip ?
 * @example setTooltipXVisible('1', 'True');
 */
function setTooltipXVisible(id, visible){
    checkWrapper("setTooltipXVisible", function () {
        var b = externalinterface.SetTooltipXVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Y property visible or not for a visualization.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Y property in tooltip ?
 * @example setTooltipYVisible('1', 'True');
 */
function setTooltipYVisible(id, visible){
    checkWrapper("setTooltipYVisible", function () {
        var b = externalinterface.SetTooltipYVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Z property visible or not for a visualization.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Z property in tooltip ?
 * @example setTooltipZVisible('1', 'True');
 */
function setTooltipZVisible(id, visible){
    checkWrapper("setTooltipZVisible", function () {
        var b = externalinterface.SetTooltipZVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Line by property visible or not for a visualization Line Chart.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Line by property in tooltip ?
 * @example setTooltipLineByVisible('1', 'True');
 */
function setTooltipLineByVisible(id, visible){
    checkWrapper("setTooltipLineByVisible", function () {
        var b = externalinterface.SetTooltipLineByVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Ratio property visible or not for a visualization Pie Chart.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Ratio property in tooltip ?
 * @example setTooltipRatioVisible('1', 'True');
 */
function setTooltipRatioVisible(id, visible){
    checkWrapper("setTooltipRatioVisible", function () {
        var b = externalinterface.SetTooltipRatioVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Sector Size property visible or not for a visualization Pie Chart.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Sector Size property in tooltip ?
 * @example setTooltipSectorSizeVisible('1', 'True');
 */
function setTooltipSectorSizeVisible(id, visible){
    checkWrapper("setTooltipSectorSizeVisible", function () {
        var b = externalinterface.SetTooltipSectorSizeVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Hierarchy property visible or not for a visualization Treemap.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Hierarchy property in tooltip ?
 * @example setTooltipHierarchyVisible('1', 'True');
 */
function setTooltipHierarchyVisible(id, visible){
    checkWrapper("setTooltipHierarchyVisible", function () {
        var b = externalinterface.SetTooltipHierarchyVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets tooltip Statistical Measures property visible or not for a visualization Box Plot.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {boolean} visible Show Statistical Measures property in tooltip ?
 * @example setTooltipStatisticalMeasuresVisible('1', 'True');
 */
function setTooltipStatisticalMeasuresVisible(id, visible){
    checkWrapper("setTooltipStatisticalMeasuresVisible", function () {
        var b = externalinterface.SetTooltipStatisticalMeasuresVisible(id, visible);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a tooltip for a visualization.
 *
 * @memberOf ibco Spotfire Visualization Properties - Tooltip
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} column The Data Table column to add in tooltip.
 * @param {string} displayname The name to display in tooltip for this column.
 * @param {string} renderer The renderer to use for this column, can be "Bitmap", "Default", "Geometry", "Image from URL", "Link" or "Virtual value".
 * @param {string} size The image size in pixel, only used when the value renderer produces images. Allowed values are 1-1600 pixels.
 */
function addTooltip(id, column, displayname, renderer, size){
    checkWrapper("addTooltip", function () {
        var b = externalinterface.AddTooltip(id, column, displayname, renderer, size);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Displays a TIBCO Spotfire visualization (no more modification will be allowed on this visualization).
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @example showvisual('1');
 */
function showvisual(id){
    checkWrapper("showvisual", function () {
        var b = externalinterface.ShowVisual(id);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Undisplays a TIBCO Spotfire visualization.
 *
 * @memberOf TIBCO Spotfire Visualization
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @example closevisual('1');
 */
function closevisual(id){
    checkWrapper("closevisual", function () {
        var b = externalinterface.CloseVisual(id);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets an automated tile mode to a TIBCO Spotfire page.
 *
 * @memberOf TIBCO Spotfire Visualization Configuration
 * @param {string} page_name The name of the page. If empty, the Spotfire active Page is used.
 * @param {string} tile_mode The tile mode: Evenly, Side-by-side, Stacked or Maximize-active.
 */
function setTileMode(page_name, tile_mode){
    checkWrapper("setTileMode", function () {
        var b = externalinterface.SetTileMode(page_name, tile_mode);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Sets a custom tile to a TIBCO Spotfire page.
 *
 * @memberOf TIBCO Spotfire Visualization Configuration
 * @param {string} tiles The Xml representing tiles: example :
    <Tile type="Horizontal" page="MyPageName">
        <Tile type="Vertical" page="MyPageName">
            <Visu id="1" />
            <Visu id="2" />
        </Tile>
        <Visu id="3" />
    </Tile>"
 * @exception If the page does not exist in the document. 
 */
function setCustomTiles(tiles){
    checkWrapper("setCustomTiles", function () {
        var b = externalinterface.SetCustomTiles(tiles);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Creates a data relation including two TIBCO Spotfire Data Tables.
 * If it already exists a relation between those two TIBCO Spotfire Data Tables, the new relation will overwrite the old one.
 *
 * @memberOf Data Table Manipulation
 * @param {string} lefttablename The name of the left data table.
 * @param {string} righttablename The name of the right data table.
 * @param {string} leftexpression Escaped column of left data table to use to create the relation. The left data table must be re-precised (see the following example).
 * @param {string} rightexpression Escaped column of right data table to use to create the relation. The right data table must be re-precised (see the following example).
 * @exception If the left data table does not exist in the document.
 * @exception If the right data table does not exist in the document.
 * @example createDataRelation('rightDT', 'leftDT', '[rightDT].[columnX]', '[leftDT].[columnY]');
 */
function createDataRelation(lefttablename, righttablename, leftexpression, rightexpression){
    checkWrapper("createDataRelation", function () {
        var b = externalinterface.CreateDataRelation(lefttablename, righttablename, leftexpression, rightexpression);
        if(!b) { throw new Error(getError()); }
    });
}

/**
 * Gets library folder files list.
 *
 * @memberOf Utils
 * @param {string} foldername The library folder name.
 * @param {string} delimiter The Files list separator.
 */
function getLibraryFolderFiles(foldername, delimiter){
    var files;
    checkWrapper("getLibraryFolderFiles", function () {
        files = externalinterface.GetLibraryFolderFiles(foldername, delimiter);
        var errormsg = getError();
        if(files == "" && errormsg != "") { throw new Error(errormsg); }
    });
    return files;
}

/**
 * Adds TIBCO Spotfire Data Rows from a URL to a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} js_url The url of the file to load.
 * @param {string} datatablename The name of the data table to which the columns are added. If empty, the Spotfire active data table is used.
 * @exception If data cannot be getted from the URL. 
 */
function addRowsFromFile(js_url, datatablename) {
    checkWrapper("addRowsFromFile", function () {
        var b=externalinterface.AddRowsFromURL(datatablename, js_url);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds TIBCO Spotfire Data Rows from an existing TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} sourcedatatablename The name of the source data table.
 * @param {string} destinationdatatablename The name of the destination data table. If empty, the Spotfire active data table is used.
 * @exception If the data table does not exist in the document. 
 */
function addRowsFromDataTable(sourcedatatablename, destinationdatatablename) {
    checkWrapper("addRowsFromDataTable", function () {
        var b=externalinterface.AddRowsFromDataTable(sourcedatatablename, destinationdatatablename);
        if (!b) { throw new Error(getError()); }
    });
}
/**
 * Adds a TIBCO Spotfire Data rows from an information link to a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of the data table to which the rows are added. If empty, the Spotfire active data table is used.
 * @param {string} informationlinkname The name of the Information Link.
 * @param {string} parameters The parameters string (i.e. ParameterName=value; for a single string value, or ParameterName={value1,value2}; for an array, or
 * ParameterName=0; for a numeric value, or ParameterName1=value1;ParameterName2={value2,value3}; for several parameters).
 * The characters "=", ";", "{", "}", "," have to be escaped by "\\" in a parameter name or in a value.
 * @exception If the information link does not exist in the library.
 * @example addRowsFromInformationLink("/myLibrary/myDirectory/myInformationLink", "Pipeline Pilot Data 1", "id=idnumber;");
 * @example addRowsFromInformationLink("/myLibrary/myDirectory/myInformationLink", "Pipeline Pilot Data 1", "id=idnumber;val=a\\,value\\{with\\=escaped\\}characters\\;;");
 */
function addRowsFromInformationLink(informationlinkname, datatablename, parameters) {

    checkWrapper("addRowsFromInformationLink", function () {
        var b=externalinterface.AddRowsFromInformationLink(informationlinkname, datatablename, parameters);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds TIBCO Spotfire Data Columns from an existing TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} sourcedatatablename The name of the source data table.
 * @param {string} destinationdatatablename The name of the destination data table.
 * @param {string} js_destinationmatchCol The name of the matching identifier data column in destination data table.
 * @param {string} js_sourcematchCol The name of the matching identifier data column in source data table.
 * @param {string} js_deleteCols The list of data columns to replace or to add. The names are sperated by a delimiter (defined in js_delimiter parameter).
 * @param {string} js_delimiter The delimiter of the columns names list.
 * @exception If the source data table does not exist in the document.
 * @exception If the destination data table does not exist in the document.
 * @exception If the column does not exist in the source data table .
 * @exception If the column does not exist in the destination data table.
 */
function addColumnsFromDataTable(sourcedatatablename, destinationdatatablename, js_destinationmatchCol, js_sourcematchCol, js_deleteCols, js_delimiter)
{
    checkWrapper("addColumnsFromDataTable", function () {
        var b=externalinterface.AddColumnFromDataTable(sourcedatatablename, destinationdatatablename, js_sourcematchCol, js_destinationmatchCol, js_deleteCols, js_delimiter);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Deletes TIBCO Spotfire Data Rows from an existing TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of the data table. If empty, the Spotfire active data table is used.
 * @param {string} columnname The name of the column containing values.
 * @param {string} values List of values indentifying rows to delete seperated by the delimiter.
 * @param {string} delimiter The values list separator.
 * @exception If the data table does not exist in the document. 
 * @exception If the column does not exist in the data table. 
 */
function deleteRows(datatablename, columnname, values, delimiter) {
    checkWrapper("deleteRows", function () {
        var b=externalinterface.DeleteRows(datatablename, columnname, values, delimiter);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Get data from a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of the data table. If empty, the Spotfire active data table is used.
 * @param {string} markingname (optional) The name of the marking.
 * @param {integer} fromrowindex (optional) The first row index of the data range.
 * @param {integer} torowindex (optional) The last row index of the data range.
 * @param {object} columns (optional) The name of the columns to get from data table. It can be a js array or just an string separated by js_delimiter.
 * @param {string} js_delimiter (optional) The delimiter of the columns names list.
 * @return {string} data in STDF format.
 * @exception If the data table does not exist in the document.
 * @exception If the marking does not exist in the document.
 * @exception If the range of rows is not valid. 
 */
function getDataTableData(datatablename, markingname, fromrowindex, torowindex, columns, js_delimiter ) {
    var data;
    checkWrapper("getDataTableData", function () {
        if (typeof columns !== "undefined")
        {
            var columnsArray = "";
            if(columns.constructor === Array)
            {
                var arrayLength = columns.length;
                js_delimiter = ",";
            }
            else
            {
                columnsArray = columns;
            }
            data = externalinterface.GetDataTableData(datatablename, markingname, fromrowindex, torowindex, columnsArray, js_delimiter);
        }
        else
        {
            data = externalinterface.GetDataTableData(datatablename, markingname, fromrowindex, torowindex, "", "");
        }
        var errormsg = getError();
        if(data == "" && errormsg != "") { throw new Error(errormsg); }
    });
    return data;
}

/**
 * Change a TIBCO Spotfire Page Title.
 *
 * @memberOf Document Manipulation
 * @param {string} oldpagetitle The title of page to rename. If empty, the Spotfire active Page is used.
 * @param {string} newpagetitle The new title of the page.
 * @exception If the page to rename does not exist in the document.
 * @exception If the new title is already used in the document.
 */
function renamePage(oldpagetitle, newpagetitle) {
    checkWrapper("renamePage", function () {
        var b = externalinterface.RenamePage(oldpagetitle, newpagetitle);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Delete a TIBCO Spotfire Page.
 *
 * @memberOf Document Manipulation
 * @param {string} pagetitle The title of page to delete. If empty, the Spotfire active Page is used.
 * @exception If the page does not exist in the document.
 */
function deletePage(pagetitle) {
    checkWrapper("deletePage", function () {
        var b = externalinterface.DeletePage(pagetitle);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Upload All data of a Tibco Spotfire Data Table to Pipeline Pilot.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of data table to upload. If empty, the Spotfire active data table is used.
 * @param {string} urlServerRoot The root URL of the Pipeline Pilot Server.
 * @param {string} sessionID The Pipeline Pilot session ID.
 * @param {string} destinationpath The destination path where data will be exported into.
 * @param {string} filename The name of uploaded file.
 * @param {string} fileformat The format of the uploaded file.
 * @param {string} columns The name of the columns to upload separated by comma(,).
 * @exception If the data table does not exist in the document.
 * @exception If the columns does not exist in the data table. 
 * @example uploadAllData("Pipeline Pilot Data 1","" ,"","/apps/discngine/", "Pipeline_Pilot_Data_1", "SBDF");
 */
function uploadAllData(datatablename, urlServerRoot, sessionID, destinationpath, filename, fileformat, columns) {
    checkWrapper("uploadAllData", function () {
        var b;
        var format;
        if (typeof fileformat !== "undefined")
        {
            format = fileformat;
        }
        else
        {
            format = "";
        }

        if (typeof columns !== "undefined")
        {

            b = externalinterface.UploadAllData(datatablename, urlServerRoot, sessionID, destinationpath, filename, format, columns );

        }
        else
        {
            b = externalinterface.UploadAllData(datatablename, urlServerRoot, sessionID, destinationpath, filename, format, "" );
        }
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Upload filtered data of a Tibco Spotfire Data Table to Pipeline Pilot.
 *
 * @memberOf Filters
 * @param {string} datatablename The name of data table to upload. If empty, the Spotfire active data table is used.
 * @param {string} urlServerRoot The root URL of the Pipeline Pilot Server.
 * @param {string} sessionID The Pipeline Pilot session ID.
 * @param {string} destinationpath The destination path where data will be exported into.
 * @param {string} filename The name of upload file.
 * @param {string} fileformat The format of the uploaded file.
 * @param {string} columns The name of the columns to upload separated by comma(,).
 * @exception If the data table does not exist in the document.
 * @exception If the columns does not exist in the data table. 
 */
function uploadFilteredData(datatablename, urlServerRoot, sessionID, destinationpath, filename, fileformat, columns) {
    checkWrapper("uploadFilteredData", function () {
        var b;
        var format;
        if (typeof fileformat !== "undefined")
        {
            format = fileformat;
        }
        else
        {
            format = "";
        }
        if (typeof columns !== "undefined")
        {

            b = externalinterface.UploadFilteredData(datatablename, urlServerRoot, sessionID, destinationpath, filename, format, columns );

        }
        else
        {
            b = externalinterface.UploadFilteredData(datatablename, urlServerRoot, sessionID, destinationpath, filename, format, "" );
        }
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Upload marked data of a Tibco Spotfire Data Table.
 *
 * @memberOf Marking
 * @param {string} datatablename The name of data table to upload. If empty, the Spotfire active data table is used.
 * @param {string} urlServerRoot The root URL of the Pipeline Pilot Server.
 * @param {string} sessionID The Pipeline Pilot session ID.
 * @param {string} destinationpath The destination path where data will be uploaded into.
 * @param {string} filename The name of uploaded file.
 * @param {string} fileformat The format of the uploaded file.
 * @param {string} marking The name of the marking to upload. If empty, the Spotfire active marking is used.
 * @param {string} columns The name of the columns to upload separated by comma(,).
 * @exception If the data table does not exist in the document.
 * @exception If the marking does not exist in the document.
 * @exception If the columns does not exist in the data table. 
 */
function uploadMarkedData(datatablename, urlServerRoot, sessionID, destinationpath, filename, fileformat, marking, columns) {
    checkWrapper("uploadMarkedData", function () {
        var b;
        var format;
        if (typeof fileformat !== "undefined")
        {
            format = fileformat;
        }
        else
        {
            format = "";
        }
        if (typeof columns !== "undefined")
        {

            b = externalinterface.UploadMarkedData(datatablename, urlServerRoot, sessionID, destinationpath, filename, format, marking, columns );

        }
        else
        {
            b = externalinterface.UploadMarkedData(datatablename, urlServerRoot, sessionID, destinationpath, filename, format, marking,"" );
        }
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Export All data of a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatablename The name of data table to export. If empty, the Spotfire active data table is used.
 * @param {string} folderpath The UNC where data will be exported into.
 * @param {string} filename The name of export file.
 * @param {string} fileformat The format of the export format.
 * @param {string} columns The name of the columns to export separated by comma(,).
 * @param {string} numRows The number of rows that will be exported.
 * @exception If the data table does not exist in the document.
 * @exception If the folder cannot be reached. 
 * @exception If the columns does not exist in the data table. 
 * @example exportAllData("Pipeline Pilot Data 1", "c:\\temp", "Pipeline_Pilot_Data_1", "STDF","", "10");
 */
function exportAllData(datatablename, folderpath, filename, fileformat, columns, numRows) {
    checkWrapper("exportAllData", function () {
        var b;
        if (typeof columns !== "undefined")
        {
            if(typeof numRows !== "undefined")
            {
                b = externalinterface.ExportAllData(datatablename, folderpath, filename, fileformat, columns,numRows);
            }
            else
            {
                b = externalinterface.ExportAllData(datatablename, folderpath, filename, fileformat, columns,"");
            }
        }
        else
        {
            b = externalinterface.ExportAllData(datatablename, folderpath, filename, fileformat, "","");
        }
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Export filtered data of a TIBCO Spotfire Data Table.
 *
 * @memberOf Filters
 * @param {string} datatablename The name of data table to export. If empty, the Spotfire active data table is used.
 * @param {string} folderpath The UNC where data will be exported into.
 * @param {string} filename The name of export file.
 * @param {string} fileformat The name of .
 * @param {string} columns The name of the columns to export separated by comma(,).
 * @param {string} numRows The number of rows that will be exported.
 * @exception If the data table does not exist in the document.
 * @exception If the folder cannot be reached. 
 * @exception If the columns does not exist in the data table. 
 */
function exportFilteredData(datatablename, folderpath, filename, fileformat, columns,numRows) {
    checkWrapper("exportFilteredData", function () {
        var b;
        if (typeof columns !== "undefined")
        {
            if(typeof numRows !== "undefined")
            {
                b = externalinterface.ExportFilteredData(datatablename, folderpath, filename, fileformat, columns,numRows);
            }
            else
            {
                b = externalinterface.ExportFilteredData(datatablename, folderpath, filename, fileformat, columns,"");
            }
        }
        else
        {
            b = externalinterface.ExportFilteredData(datatablename, folderpath, filename, fileformat, "","");
        }
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Export marked data of a TIBCO Spotfire Data Table.
 *
 * @memberOf Marking
 * @param {string} datatablename The name of data table to export. If empty, the Spotfire active data table is used.
 * @param {string} folderpath The UNC where data will be exported into.
 * @param {string} filename The name of export file.
 * @param {string} fileformat The format of the export format.
 * @param {string} marking The name of the marking to export. If empty, the Spotfire active marking is used.
 * @param {string} columns The name of the columns to export separated by comma(,).
 * @param {string} numRows The number of rows that will be exported.
 * @exception If the data table does not exist in the document.
 * @exception If the marking does not exist in the document.
 * @exception If the folder cannot be reached. 
 * @exception If the columns does not exist in the data table. 
 */
function exportMarkedData(datatablename, folderpath, filename, fileformat, marking, columns, numRows) {
    checkWrapper("exportMarkedData", function () {
        var b;
        if (typeof columns !== "undefined")
        {
            if(typeof numRows !== "undefined")
            {
                b = externalinterface.ExportMarkedData(datatablename, folderpath, filename, fileformat, marking, columns, numRows);
            }
            else
            {
                b = externalinterface.ExportMarkedData(datatablename, folderpath, filename, fileformat, marking, columns, "");
            }
        }
        else
        {
            b = externalinterface.ExportMarkedData(datatablename, folderpath, filename, fileformat, marking, "", "");
        }
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Gets the type of the filter applied on the provided column.
 *
 * @memberOf Filters
 * @param {string} columnname Name of the column on which create the filter.
 * @return {string} the display name of the filter type. Null if a problem occur during the process.
 */
function getFilterType(columnname)
{
    var result;
    checkWrapper("getFilterType", function () {
        result = externalinterface.GetFilterType(columnname);
    });
    return result;
}

/**
 * Gets checked values in the specified CheckBox Filter.
 *
 * @memberOf Filters
 * @param {string} columnname Name of the column on which create the filter.
 * @param {string} delimiter Delimiter to use to separate checked values.
 * @return {string} Values checked by the CheckBox Filter. Null if a problem occur during the process.
 */
function getCheckBoxFilterCheckedValues(columnname, delimiter)
{
    var result;
    checkWrapper("getCheckBoxFilterCheckedValues", function () {
        result = externalinterface.GetCheckBoxFilterCheckedValues(columnname, delimiter);
    });
    return result;
}

/**
 * Gets the Low value of a Range Filter.
 *
 * @memberOf Filters
 * @param {string} columnname Name of the column associated to the filter.
 * @return {string} Low value of the Range Filter. Null if a problem occur during the process.
 * @exception If the collection of filters cannot be getted. 
 */
function getRangeFilterLowValue(columnname)
{
    var result;
    checkWrapper("getRangeFilterLowValue", function () {
        result = externalinterface.GetRangeFilterLowValue(columnname);
    });
    return result;
}

/**
 * Gets the High value of a Range Filter.
 *
 * @memberOf Filters
 * @param {string} columnname Name of the column associated to the filter.
 * @return {string} High value of the Range Filter. Null if a problem occur during the process.
 * @exception If the collection of filters cannot be getted. 
 */
function getRangeFilterHighValue(columnname)
{
    var result;
    checkWrapper("getRangeFilterHighValue", function () {
        result = externalinterface.GetRangeFilterHighValue(columnname);
    });
    return result;
}

/**
 * Sets the values checked by the specified CheckBox Filter. A reset is performed before setting values.
 *
 * @memberOf Filters
 * @param {string} columnname Name of the column on which create the filter.
 * @param {string} values List of values to checked delimited with "delimiter".
 * @param {string} delimiter Delimiter used in the list of values.
 * @param {string} keepemptychecked Optional boolean to set the "include empty value" value (false to uncheck it).
 * @exception If the targeted filter is not a check box filter. 
 */
function setCheckBoxFilterCheckedValues(columnname, values, delimiter, keepemptychecked)
{
    checkWrapper("setCheckBoxFilterCheckedValues", function () {
        var b=externalinterface.SetCheckBoxFilterCheckedValues(columnname, values, delimiter, keepemptychecked);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the value range of the specified Range Filter. A reset is performed before setting values.
 *
 * @memberOf Filters
 * @param {string} columnname Name of the column on which create the filter.
 * @param {string} values The two values to use for the value range, delimited with "delimiter".
 * @param {string} delimiter Delimiter used in the list of values.
 * @param {string} keepemptychecked Optional boolean to set the "include empty value" value (false to uncheck it).
 * @exception If the targeted filter is not a range filter. 
 */
function setRangeFilterValueRange(columnname, values, delimiter, keepemptychecked)
{
    checkWrapper("setRangeFilterValueRange", function () {
        var b=externalinterface.SetRangeFilterValueRange(columnname, values, delimiter, keepemptychecked);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Resets the specified CheckBox Filter.
 *
 * @memberOf Filters
 * @param {string} columnname Name of the column on which create the filter.
 */
function resetCheckBoxFilterCheckedValues(columnname)
{
    checkWrapper("resetCheckBoxFilterCheckedValues", function () {
        var b=externalinterface.ResetCheckBoxFilterCheckedValues(columnname);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Resets the specified Range Filter.
 *
 * @memberOf Filters
 * @param {string} columnname Name of the column on which create the filter.
 */
function resetRangeFilterValueRange(columnname)
{
    checkWrapper("resetRangeFilterValueRange", function () {
        var b=externalinterface.ResetRangeFilterValueRange(columnname);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Resets all filters of the active Filtering Scheme in the document.
 *
 * @memberOf Filters
 */
function resetAllFilters()
{
    checkWrapper("resetAllFilters", function () {
        var b=externalinterface.ResetAllFilters();
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Resets all visible filters of the active Filtering Scheme in the document.
 *
 * @memberOf Filters
 */
function resetAllVisibleFilters()
{
    checkWrapper("resetAllVisibleFilters", function () {
        var b=externalinterface.ResetAllVisibleFilters();
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Marks filtered rows in the document.
 *
 * @memberOf Marking
 */
function markFilteredRows()
{
    checkWrapper("markFilteredRows", function () {
        var b=externalinterface.MarkFilteredRows();
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Unmarks rows marked by the active marking in the active Data Table.
 *
 * @memberOf Marking
 */
function unmarkMarkedRows()
{
    checkWrapper("unmarkMarkedRows", function () {
        var b=externalinterface.UnmarkMarkedRows();
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Invert the marking in the active Data Table, that is marked rows are unmarked and other rows are marked.
 *
 * @memberOf Marking
 */
function invertMarkedRows()
{
    checkWrapper("invertMarkedRows", function () {
        var b=externalinterface.InvertMarkedRows();
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Deletes marked rows from data table.
 *
 * @memberOf Marking
 */
function deleteMarkedRows()
{
    checkWrapper("deleteMarkedRows", function () {
        var b=externalinterface.DeleteMarkedRows();
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Gets the list of TIBCO Spotfire Page names.
 *
 * @memberOf Document Manipulation
 * @param {string} delimiter Delimiter of the list of page names.
 * @return {string} A list of Spotfire Page Names separated by the specified delimiter.
 */
function getPages(delimiter) {
    var pages;
    checkWrapper("getPages", function () {
        pages = externalinterface.GetPages(delimiter);
        var errormsg = getError();
        if (pages == "" && errormsg != "") { throw new Error(errormsg); }
    });
    return pages;
}

/**
 * Gets the active TIBCO Spotfire Page name.
 *
 * @memberOf Document Manipulation
 * @return {string} The name of the active Spotfire Page.
 */
function getActivePage() {
    var page;
    checkWrapper("getActivePage", function () {
        page = externalinterface.getActivePage();
        var errormsg = getError();
        if (page == "" && errormsg!= "") { throw new Error(errormsg); }
    });
    return page;
}

/**
 * Sets Page as active in the TIBCO Spotfire Document.
 *
 * @memberOf Document Manipulation
 * @param {string} pageName Name of the page to set as active.
 * @exception If the page does not exist in the document. 
 */
function setActivePage(pageName)
{
    checkWrapper("setActivePage", function () {
        var b=externalinterface.SetActivePage(pageName);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Gets the active TIBCO Spotfire Data Table name.
 *
 * @memberOf Data Table Manipulation
 * @return {string} The name of the active TIBCO Spotfire Data Table.
 */
function getActiveDataTable() {
    var b;
    checkWrapper("getActiveDataTable", function () {
        b = externalinterface.GetActiveDataTable();
        var errormsg = getError();
        if (errormsg!= "") { throw new Error(errormsg); }
    });
    return b;
}

/**
 * Gets the default TIBCO Spotfire Data Table name.
 *
 * @memberOf Data Table Manipulation
 * @return {string} The name of the default TIBCO Spotfire Data Table.
 */
function getDefaultDataTable() {
    var b;
    checkWrapper("getDefaultDataTable", function () {
        b = externalinterface.GetDefaultDataTable();
        var errormsg = getError();
        if (errormsg != "") { throw new Error(errormsg); }
    });
    return b;
}

/**
 * Sets Data Table as default in the TIBCO Spotfire Document.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatableName Name of the Data Table to set as default.
 * @exception If the data table does not exist in the document. 
 */
function setDefaultDataTable(datatableName)
{
    checkWrapper("setDefaultDataTable", function () {
        var b = externalinterface.SetDefaultDataTable(datatableName);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Changes the TIBCO Spotfire Data Table of a visualization.
 *
 * @memberOf Data Table Manipulation
 * @param {string} id The id of the visualization. Unique in the current TIBCO Spotfire Document.
 * @param {string} datatable The name of TIBCO Spotfire Data table to set.
 */
function changeDataTable(id, datatable) {
    var b;
    checkWrapper("changeDataTable", function () {
        b = externalinterface.ChangeTable(id, datatable);
    });
    return b;
}

/**
 * Changes the TIBCO Spotfire Data Table of all visualizations.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatable The name of TIBCO Spotfire Data table to set.
 */
function changeDataTableForAll(datatable) {
    var b;
    checkWrapper("changeDataTableForAll", function () {
        b = externalinterface.ChangeTableForAll(datatable);
    });
    return b;
}

/**
 * Delete all rows from a TIBCO Spotfire Data Table.
 *
 * @memberOf Data Table Manipulation
 * @param {string} datatable The name of the TIBCO Spotfire Data table to clear. If empty, the Spotfire active data table is used.
 * @exception If the data table does not exist in the document.
 */
function deleteAllRows(datatable) {
    var b;
    checkWrapper("deleteAllRows", function () {
        b = externalinterface.DeleteAllRows(datatable);
    });
    return b;
}

/**
 * Refresh all Data Tables on the current Document.
 *
 * @memberOf Data Table Manipulation
 */
function refreshAllDataTables() {
    var refreshed;
    checkWrapper("refreshAllDataTables", function () {
        refreshed = externalinterface.RefreshAllDataTables();            
        var errormsg = getError();
        if ( errormsg != "" ) { throw new Error(errormsg); }
    });
    return refreshed;
}

/**
 * Adds an horizontal line to a Spotfire visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {string} datatable Name of the Spotfire Data Table. Can be empty.
 * @param {string} expression Expression of the horizontal line to draw.
 * If the "datatable" is null or empty, the "expression" is either a
 * fixed value or an aggregated value (Min, Max, ...) on the Y axis or a custom expression.
 * If the "datatable" is precised, the "expression" must indicate a column name of the "datatable".
 * @example addHorizontalLine("1", "1", "", "Max(Y)");
 * @example addHorizontalLine("1", "1", "Pipeline Pilot Data 1", "CL_VAR");
 */
function addHorizontalLine(id, idLine, datatable, expression)
{
    checkWrapper("addHorizontalLine", function () {
        var b = externalinterface.AddHorizontalLine(id, idLine, datatable, expression);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a vertical line to a Spotfire visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {string} datatable Name of the Spotfire Data Table. Can be empty.
 * @param {string} expression Expression of the vertical line to draw.
 * If the "datatable" is null or empty, the "expression" is either a
 * fixed value or an aggregated value (Min, Max, ...) on the X axis or a custom expression.
 * If the "datatable" is precised, the "expression" must indicate a column name of the "datatable".
 * @example addVerticalLine("1", "2", "", "Max(X)");
 * @example addVerticalLine("1", "2", "Pipeline Pilot Data 1", "CL_VAR");
 */
function addVerticalLine(id, idLine, datatable, expression)
{
    checkWrapper("addVerticalLine", function () {
        var b = externalinterface.AddVerticalLine(id, idLine, datatable, expression);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a curve to a Spotfire visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {string} datatable Name of the Spotfire Data Table. If it is empty, this function corresponds to the "Curve Draw..." option in Spotfire,
 * if it is precised, this function corresponds to the "Curve From Data Table..." option.
 * @param {string} expression Expression of the curve to draw.
 * @example addCurve(id, "1", "", "5.0*Sin(x) + 2.5");
 * @example addCurve(id, "1", "Pipeline Pilot Data 1", "5.0*Sin(x) + 2.5");
 */
function addCurve(id, idLine, datatable, expression)
{
    checkWrapper("addCurve", function () {
        var b = externalinterface.AddCurve(id, idLine, datatable, expression);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a straight line that minimizes the sum-of-squares of the y-distances between the data and the line.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @example addStraightLineFit("1", "3");
 */
function addStraightLineFit(id, idLine)
{
    checkWrapper("addStraightLineFit", function () {
        var b=externalinterface.AddStraightLineFit(id, idLine);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a line or multiline. The points of the line are defined from the values of a pair of columns.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {string} datatable Name of the Spotfire Data Table containing the columns.
 * If empty, the datatable associated to the visualization is used.
 * @param {string} xValuesColumn Name of the column containing the x-values.
 * @param {string} yValuesColumn Name of the column containing the y-values.
 * @example addColumnValuesLine("1", "4", "Pipeline Pilot Data 1", "ALogP", "Molecular_Weight");
 * @example addColumnValuesLine("1", "4", "", "ALogP", "CL_SIZE");
 */
function addColumnValuesLine(id, idLine, datatable, xValuesColumn, yValuesColumn)
{
    checkWrapper("addColumnValuesLine", function () {
        var b=externalinterface.AddColumnValuesLine(id, idLine, datatable, xValuesColumn, yValuesColumn);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a polynomial curve fit.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {integer} degree Degree of the polynomial.
 * @example addPolynomialCurve("1", "3", 5);
 */
function addPolynomialCurve(id, idLine, degree)
{
    checkWrapper("addPolynomialCurve", function () {
        var b=externalinterface.AddPolynomialCurve(id, idLine, degree);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a logistic regression curve fit.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {double} min The fixed min. Leave empty for automatic calculation.
 * @param {double} max The fixed min. Leave empty for automatic calculation.
 * @param {integer} XTransformType X-axis transformation type : 0 for None, 1 for log10.
 * @example addLogisticRegressionCurve("1", "1", "", "", 1);
 */
function addLogisticRegressionCurve(id, idLine, min, max, XTransformType)
{
    checkWrapper("addLogisticRegressionCurve", function () {
        var b=externalinterface.AddLogisticRegressionCurve(id, idLine, min, max, XTransformType);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a power curve fit.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @example addPowerCurve("1", "3");
 */
function addPowerCurve(id, idLine)
{
    checkWrapper("addPowerCurve", function () {
        var b=externalinterface.AddPowerCurve(id, idLine);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a logarithmic curve fit.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @example addLogarithmicCurve("1", "3");
 */
function addLogarithmicCurve(id, idLine)
{
    checkWrapper("addLogarithmicCurve", function () {
        var b=externalinterface.AddLogarithmicCurve(id, idLine);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds an exponential curve fit.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @example addExponentialCurve("1", "3");
 */
function addExponentialCurve(id, idLine)
{
    checkWrapper("addExponentialCurve", function () {
        var b=externalinterface.AddExponentialCurve(id, idLine);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a gaussian curve fit.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {double} amplitude The amplitude of the Gaussian curve. Leave empty for automatic calculation.
 * @param {double} position The position (mean) of the Gaussian curve. Leave empty for automatic calculation.
 * @param {double} width The width of the Gaussian curve. Leave empty for automatic calculation.
 * @example addGaussianCurve("1", "3", "", "", 10.0, true, "", "");
 */
function addGaussianCurve(id, idLine, amplitude, position, width)
{
    checkWrapper("addGaussianCurve", function () {
        var b=externalinterface.AddGaussianCurve(id, idLine, amplitude, position, width);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds the "Sorted by column" property to a multiline/curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {string} sortedBy Name of the column used to sort the x- and y-values before connecting the points to a multiline.
 * @example linesAndCurvesSortedByColumn("1", "4", "ALogP");
 */
function linesAndCurvesSortedByColumn(id, idLine, sortedBy)
{
    checkWrapper("linesAndCurvesSortedByColumn", function () {
        var b=externalinterface.LinesAndCurvesSortedByColumn(id, idLine, sortedBy);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the "Empty values" property of a line/curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {boolean} ignoreEmpty Ignore (skip) empty values? If false, splits line at empty values.
 * @example linesAndCurvesEmptyValues("1", "4", true);
 */
function linesAndCurvesEmptyValues(id, idLine, ignoreEmpty)
{
    checkWrapper("linesAndCurvesEmptyValues", function () {
        var b=externalinterface.LinesAndCurvesEmptyValues(id, idLine, ignoreEmpty);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the "Curve name" property of a line/curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {string} customName Name of the line.
 * @example linesAndCurvesCustomName("1", "4", "LineOnALogP");
 */
function linesAndCurvesCustomName(id, idLine, customName)
{
    checkWrapper("linesAndCurvesCustomName", function () {
        var b=externalinterface.LinesAndCurvesCustomName(id, idLine, customName);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the appearance properties of a line/curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {string} color The color, either in the HTML format or its name, by which the line is rendered.
 * Can be empty, default is black.
 * @param {integer} transparency The transparency of the line, between 0 and 255, default is 255 (no transparency).
 * @param {integer} lineStyle Code indicating the line style by which the line is rendered: 0 for dash line, 1 for dot line,
 * 2 for single line, 3 for double line. Can be empty, default is single line.
 * @param {integer} width The width of the line. Can be empty, default is 2.
 * @param {boolean} isBackground Value controlling if the line is renderer behind other visual elements in the visualization.
 * Can be empty, default is "False".
 * @example linesAndCurvesSetAppearance("1", "4", "#FF0000", 50, 3, 1, true);
 */
function linesAndCurvesSetAppearance(id, idLine, color, transparency, lineStyle, width, isBackground)
{
    checkWrapper("linesAndCurvesSetAppearance", function () {
        var b=externalinterface.LinesAndCurvesSetAppearance(id, idLine, color, transparency, lineStyle, width, isBackground);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the inflection point appearance properties of a logistic regression curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {string} color The color, either in the HTML format or its name, by which the line is rendered.
 * Can be empty, default is black.
 * @param {integer} transparency The transparency of the line, between 0 and 255. Can be empty, default is 255 (no transparency).
 * @param {integer} shape The code indicating the shape to use for the point. Can be empty, default is Circle.
 * Codes: 0: Circle, 1: CircleOutline, 2: Cross, 3: CrossLine, 4: CrossOutline, 5: Diamond, 6: DiamondOutline, 7: DownPointingArrow, 8: DownPointingArrowLine, 9: DownPointingArrowOutline,
 * 10: HorizontalLine, 11: HorizontalRectangle, 12: HorizontalRectangleOutline, 13: LeftPointingArraw, 14: LeftPointingArrawLine, 15: LeftPointingArrawOutline,
 * 16: Pentagon, 17: PentagonOutline, 18: Plus, 19: PlusLine, 20: PlusOutline, 21: RightPointingArrow, 22: RightPointingArrowLine, 23: RightPointingArrowOutline,
 * 24: Square, 25: SquareOutline, 26: StarFive, 27: StarFiveOutline, 28: StarFour, 29: StarFourOutline, 30: StarSixLine, 31: TriangleDown, 32: TriangleDownOutline,
 * 33: TriangleLeft, 34: TriangleLeftOutline, 35: TriangleRight, 36: TriangleRightOutline, 37: TriangleUp, 38: TriangleUpOutline, 39: UpPointingArrow,
 * 40: UpPointingArrowLine, 41: UpPointingArrowOutline, 42: VerticalLine, 43: VerticalRectangle, 44: VerticalRectangleOutline.
 * @param {float} pointSize The size of the inflection point, between 0 and 100. Can be empty.
 * @param {boolean} isBackground Value controlling if the line is renderer behind other visual elements in the visualization.
 * Can be empty, default is "False".
 * @example linesAndCurvesSetInflectionPointAppearance("1", "4", "#FF0000", 100, 3, 50, false);
 */
function linesAndCurvesSetInflectionPointAppearance(id, idLine, color, transparency, shape, pointSize, isBackground)
{
    checkWrapper("linesAndCurvesSetInflectionPointAppearance", function () {
        var b=externalinterface.LinesAndCurvesSetInflectionPointAppearance(id, idLine, color, transparency, shape, pointSize, isBackground);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the "Included in axis range" property of a line/curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {boolean} affect Included in axis range ?
 * @example linesAndCurvesAffectAxisRange("1", "4", true);
 */
function linesAndCurvesAffectAxisRange(id, idLine, affect)
{
    checkWrapper("linesAndCurvesAffectAxisRange", function () {
        var b=externalinterface.LinesAndCurvesAffectAxisRange(id, idLine, affect);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the inflection point "Included in axis range" property of a logistic regression curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {boolean} affect Included in axis range ?
 * @example linesAndCurvesInflectionPointAffectAxisRange("1", "4", true);
 */
function linesAndCurvesInflectionPointAffectAxisRange(id, idLine, affect)
{
    checkWrapper("linesAndCurvesInflectionPointAffectAxisRange", function () {
        var b=externalinterface.LinesAndCurvesInflectionPointAffectAxisRange(id, idLine, affect);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the "Update manually" property of a line/curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {boolean} update Update manually ?
 * @example linesAndCurvesUpdateManually("1", "4", true);
 */
function linesAndCurvesUpdateManually(id, idLine, update)
{
    checkWrapper("linesAndCurvesUpdateManually", function () {
        var b=externalinterface.LinesAndCurvesUpdateManually(id, idLine, update);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Updates a line or curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @example linesAndCurvesUpdate("1", "4");
 */
function linesAndCurvesUpdate(id, idLine)
{
    checkWrapper("linesAndCurvesUpdate", function () {
        var b=externalinterface.LinesAndCurvesUpdate(id, idLine);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the individual fitting modes properties for a line/curve (called "One per" in the Spotfire interface).
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {integer} fittingMode Individual fitting mode codes : 0 for None, 1 for Color, 2 for Shape, 3 for Line, 4 for Trellis.
 * @example linesAndCurvesIndividualFittingModes("1", "4", 2);
 */
function linesAndCurvesIndividualFittingModes(id, idLine, fittingMode)
{
    checkWrapper("linesAndCurvesIndividualFittingModes", function () {
        var b=externalinterface.LinesAndCurvesIndividualFittingModes(id, idLine, fittingMode);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Sets the line/curve visible or not.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {boolean} visible Set the line/curve visible ?
 * @example linesAndCurvesSetVisible("1", "4", false);
 */
function linesAndCurvesSetVisible(id, idLine, visible)
{
    checkWrapper("linesAndCurvesSetVisible", function () {
        var b=externalinterface.LinesAndCurvesSetVisible(id, idLine, visible);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Adds a label and a tooltip to a line/curve.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id The visualization identifier.
 * @param {string} idLine The line/curve identifier.
 * @param {string} columnName Name of the column to add in tooltips of the line/curve.
 * @param {boolean} showInLabel Show in label of the line/curve ? Can be "null", null, or "". By default, false.
 * @param {boolean} showInTooltip Show in tooltip of the line/curve ? Can be "null", null, or "". By default, true.
 * @example linesAndCurvesAddLabelAndTooltip("1", "4", "CL_SIZE", true, true);
 * @example linesAndCurvesAddLabelAndTooltip("1", "4", "Value (x)", true, true);
 */
function linesAndCurvesAddLabelAndTooltip(id, idLine, columnName, showInLabel, showInTooltip)
{
    checkWrapper("linesAndCurvesAddLabelAndTooltip", function () {
        var b=externalinterface.LinesAndCurvesAddLabelAndTooltip(id, idLine, columnName, showInLabel, showInTooltip);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Remove the precised list of lines and curves drew on a Spotfire visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id Visualization identifier.
 * @param {string} idLinesList List of line/curve identifiers to remove from the visualization.
 * @param {string} separator Separator used in idLinesList.
 * @example closevisual("1"); clearLinesAndCurves("1", "2;5", ";"); showVisual("1");
 */
function removeLinesAndCurves(id, idLinesList, separator)
{
    checkWrapper("removeLinesAndCurves", function () {
        var b=externalinterface.RemoveLinesAndCurves(id, idLinesList, separator);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Remove all lines and curves drew on a Spotfire visualization.
 *
 * @memberOf TIBCO Spotfire Visualization Properties - Lines & Curves
 * @param {string} id Visualization identifier.
 * @example closevisual("1"); clearAllLinesAndCurves("1"); showVisual("1");
 */
function removeAllLinesAndCurves(id)
{
    checkWrapper("removeAllLinesAndCurves", function () {
        var b=externalinterface.RemoveAllLinesAndCurves(id);
        if (!b) { throw new Error(getError()); }
    });
}

/*************SCRIPT*METHODS******************************************************************************************************/
/**
 * Runs an ironPython script in the TIBCO Spotfire Client.
 * To get standard output and standard error, use getRunStdOut() and getRunStdErr() functions.
 *
 * @memberOf Script
 * @param {string} scriptCode The script body in ironPython, base 64 encoded.
 */
function runScript(scriptCode)
{
    checkWrapper("runScript", function () {
        var b=externalinterface.RunScript(scriptCode);
        if (!b) {
            throw new Error(getError()); 
        }
        
        var lastScriptError = getRunStdErr();
        if (lastScriptError !== "") {
            throw new Error(lastScriptError); 
        }
    });
}

/**
 * Runs an Operating System command line.
 * To get standard output and standard error, use getRunStdOut() and getRunStdErr() functions.
 *
 * @memberOf Script
 * @param {string} commandline Command line to execute, base 64 encoded.
 * @exception If the command line is null or empty.
 * @example runCommandLine("mspaint");
 */
function runCommandLine(commandline)
{
    checkWrapper("runCommandLine", function () {
        var b=externalinterface.RunCommandLine(commandline);
        if (!b) { throw new Error(getError()); }
    });
}

/**
 * Gets the standard output of the last command line or script running.
 *
 * @memberOf Script
 * @return The last standard output message.
 */
function getRunStdOut()
{
    var lastStdOut = "";
    checkWrapper("getRunStdOut", function () {
        lastStdOut = externalinterface.RunStdOut;
    });
    return lastStdOut;
}

/**
 * Gets the standard error of the last command line or script running.
 *
 * @memberOf Script
 * @return The last standard error message.
 */
function getRunStdErr()
{
    var lastStdErr = "";
    checkWrapper("getRunStdErr", function () {
        lastStdErr = externalinterface.RunStdErr;
    });
    return lastStdErr;
}

/*************UTILITIES*METHODS******************************************************************************************************/

/**
 * Returns the code associated to the shape name provided in argument.
 *
 * @memberOf Utils
 * @param {string} name Shape name.
 * @return {integer} The code associated to the shape name.
 */
function getIntFromShapeName(name)
{
    try {
        switch(name)
        {
            case "Circle": return 0;
            case "CircleOutline": return 1;
            case "Cross": return 2;
            case "CrossLine": return 3;
            case "CrossOutline": return 4;
            case "Diamond": return 5;
            case "DiamondOutline": return 6;
            case "DownPointingArrow": return 7;
            case "DownPointingArrowLine": return 8;
            case "DownPointingArrowOutline": return 9;
            case "HorizontalLine": return 10;
            case "HorizontalRectangle": return 11;
            case "HorizontalRectangleOutline": return 12;
            case "LeftPointingArrow": return 13;
            case "LeftPointingArrowLine": return 14;
            case "LeftPointingArrowOutline": return 15;
            case "Pentagon": return 16;
            case "PentagonOutline": return 17;
            case "Plus": return 18;
            case "PlusLine": return 19;
            case "PlusOutline": return 20;
            case "RightPointingArrow": return 21;
            case "RightPointingArrowLine": return 22;
            case "RightPointingArrowOutline": return 23;
            case "Square": return 24;
            case "SquareOutline": return 25;
            case "StarFive": return 26;
            case "StarFiveOutline": return 27;
            case "StarFour": return 28;
            case "StarFourOutline": return 29;
            case "StarSixLine": return 30;
            case "TriangleDown": return 31;
            case "TriangleDownOutline": return 32;
            case "TriangleLeft": return 33;
            case "TriangleLeftOutline": return 34;
            case "TriangleRight": return 35;
            case "TriangleUp": return 36;
            case "TriangleUpOutline": return 37;
            case "UpPointingArrow": return 38;
            case "UpPointingArrowLine": return 39;
            case "UpPointingArrowOutline": return 40;
            case "VerticalLine": return 41;
            case "VerticalRectangle": return 42;
            case "VerticalRectangleOutline": return 43;
            default: return 0;
        }
    } catch (error) {
        if(showAlerts){
            alert("Error getIntFromShapeName: " + error.message);
        }
        else{
            throw new Error("Error getIntFromShapeName: " + error.message);
        }
    }
}