var dng = (function(dng) {
    return dng ;
}(dng || {})
) ;

dng.tsc = (function(my) {

  var _COLUMNS_SEPARATOR   = "!!_!!";
  var _SYN_DOC_PROP_PREFIX = "PPSYNC";
  var _ROWID_COL           = "TSC_ROWID"  ;
  var _MODE_OVERWRITE      = "Overwrite"  ;
  var _MODE_MERGE          = "Merge"  ; 

/*************************************************************************
            ------------ Synchronization -------------
***************************************************************************/

  my.sync = function(pSessionID, pParams) {
    if(typeof pSessionID === 'undefined' || pSessionID == '') {
      throw 'Session ID is undefined';
    }
    
    return new Sync(pSessionID, pParams);
  };
  
  function Sync(pSessionID, pParams) { 
    this.sessionID  = pSessionID ;
    this.dataSender = null ;
    
    
    pParams = typeof pParams === 'undefined' || !pParams || typeof pParams !== 'object' ? {} : pParams;

    this.params = {
      syncFiltered   : pParams.syncFiltered   || false,
      syncMarked     : pParams.syncMarked     || false,
    } ;
    
    this._callbacks = {
      syncStart     : [],
      syncNextStart : [],
      syncNextDone  : [],
      syncNextFail  : [],
      syncDone      : [],
      syncFail      : []
    } ;
    
    this._callbacksInternal = {
      syncStart     : [],
      syncNextStart : [],
      syncNextDone  : [],
      syncNextFail  : [],
      syncDone      : [],
      syncFail      : []
    } ;
    
    this._callbacksInternal.syncNextDone.push(function(syncState, dataTable) {
      syncState.i ++ ;
      syncState.dataTablesProcessed.push(dataTable);
      syncState.dataTablesMap[dataTable.name] = dataTable;
      
      this.syncNext(syncState);
    });
    
    this._callbacksInternal.syncDone.push(function(syncState) {
      syncState.running = false ;
    });
    
    this._callbacksInternal.syncFail.push(function(syncState, dataTable, errorMessage) {
      syncState.running = false ;
    });
    
    return this._prepareDocument(pSessionID);
  } ;
  
/**
 * Prepares the current Spotfire document for synchronization. This involves
 * the folowing :
 * - Reseting document properties related to synchronization if the session identifier
 *   is different from the current one (pSessionID)
 * - Setting the current session identifier document property value
 *
 */
  Sync.prototype._prepareDocument = function(pSessionID) {
    // Reset session properties if we have switched to a different session
    var oldSessionID = getDocumentProperty(dng.tsc.getPPSIDPropertyName());
    
    if(oldSessionID == '' || oldSessionID != pSessionID) {
      // Reset document properties if a different session is used for the current
      // document
      getDocumentPropertyNames('-')
        .split('-')
        .filter(function(p) {
          return p.indexOf(_SYN_DOC_PROP_PREFIX) == 0 ;
        })
        .forEach(function(p){
          setDocumentProperty(p, 'null'); 
        });
    }
		setDocumentProperty(dng.tsc.getPPSIDPropertyName(), pSessionID);
    
    return this ;
  } ;

/**
 * Defines the callback function in charge of exporting the data table (and
 * the filtered / marked rows) and importing it back to the Pipeline Pilot server.
 *
 * The function must have the following functions available:
 *
 * - syncDataTable(dataTable, onDone, onFail)
 * - syncMarkedRows(dataTable, onDone, onFail)
 * - syncFilteredRows(dataTable, onDone, onFail)
 *
 */
  Sync.prototype.setDataSender = function(f) {
    this.dataSender = f;
    // Sanity check for data sender
    if(typeof f.syncDataTable !== 'function') {
      throw "Invalid data sender: expected prototype function syncDataTable was not found";
    }
    
    if(typeof f.syncFilteredRows !== 'function') {
      throw "Invalid data sender: expected prototype function syncFilteredRows was not found";
    }

    if(typeof f.syncMarkedRows !== 'function') {
      throw "Invalid data sender: expected prototype function syncMarkedRows was not found";
    }
    
    return this ;
  } ;
  
/**
 * Register a callback for the specified synchronization event. The following events
 * are currently supported:
 * 
 * onSyncStart     : when the synchronization itself starts
 * onSyncNextStart : when the synchronization of the current data table starts
 * onSyncNextDone  : when the synchronization of the current data table ends
 * onSyncDone      : when the synchronization process is done
 * onSyncFail      : when the synchronization process failed (uncatched exception)
 *
 * Each event will trigger the call to the target callback with the following parameter
 *
 * onSyncStart     : f(sync)
 * onSyncNextStart : f(sync, dataTable)
 * onSyncNextDone  : f(sync, dataTable)
 * onSyncNextFail  : f(sync, dataTable)
 * onSyncDone      : f(sync)
 * onSyncFail      : f(sync, dataTable, errorMessage)
 *
 */  
  Sync.prototype.on = function(event, callback) {
    if (!this._callbacks.hasOwnProperty(event)) {
      throw "Unsupported synchronization event: "+event;
    }
    this._callbacks[event].push(callback);
    
    return this ;
  } ;

/**
 * Fires a onSyncStart event
 *
 */  
  Sync.prototype.fireSyncStarted = function(syncState) {
    var event = "syncStart" ;
    var that = this ;
    
    this._callbacks[event]
    .forEach(function(c) {
      c.call(that);
    });
    
    this._callbacksInternal[event]
    .forEach(function(c) {
      c.call(that, syncState);
    });
  };
  
/**
 * Fires a onSyncStart event
 *
 */  
  Sync.prototype.fireSyncNextStarted = function(syncState, dataTable) {
    var event = "syncNextStart" ;
    var that = this ;
    
    this._callbacks[event]
    .forEach(function(c) {
      c.call(that, dataTable);
    });
    
    this._callbacksInternal[event]
    .forEach(function(c) {
      c.call(that, syncState, dataTable);
    });
  };
  
/**
 * Fires a onSyncNextDone event
 *
 */  
  Sync.prototype.fireSyncNextDone = function(syncState, dataTable) {
    var event = "syncNextDone" ;
    var that = this ;
    
    this._callbacks[event]
    .forEach(function(c) {
      c.call(that, dataTable);
    });
    
    this._callbacksInternal[event]
    .forEach(function(c) {
      c.call(that, syncState, dataTable);
    });
  };
   
/**
 * Fires a onSyncDone event
 *
 */  
  Sync.prototype.fireSyncDone = function(syncState) {
    var event = "syncDone" ;
    var that = this ;
    
    this._callbacks[event]
    .forEach(function(c) {
      c.call(that, syncState);
    });
    
    this._callbacksInternal[event]
    .forEach(function(c) {
      c.call(that, syncState);
    });
  };
  
/**
 * Fires a onSyncFail event
 *
 */  
  Sync.prototype.fireSyncFailed = function(syncState, dataTable, errorMessage) {
    var event = "syncFail" ;
    var that = this ;
    
    this._callbacks[event]
    .forEach(function(c) {
      c.call(that, syncState, dataTable, errorMessage);
    });
    
    this._callbacksInternal[event]
    .forEach(function(c) {
      c.call(that, syncState, dataTable, errorMessage);
    });
  }; 
  
/**
 * Initializes a new synchronization state
 *
 */  
  Sync.prototype._initSyncState = function(pDataTables) {
    return {
      dataTables2Sync     : pDataTables.slice(),
      dataTablesProcessed : [],
      dataTablesMap       : {},
      running             : false,
      i                   : 0
    };
  } ;
  
/**
 * Start the synchronization procedure.
 *
 */  
  Sync.prototype.start = function(pDataTables) {
    if(this.running) {
      return this ;
    }
    
    var syncState = this._initSyncState(pDataTables);
	
    // Populate the list of data tables to synchonize
    this.fireSyncStarted(syncState);
	
    // Next, check if we actually need to sync those data tables
    var that = this ;
    setTimeout(function() {
      that.syncNext(syncState);
    }, 50);
    
    return this ;
  } ;
  
/**
 * Synchronize the next data table.
 *
 */  
  Sync.prototype.syncNext = function(syncState) {
    if(syncState.i >= syncState.dataTables2Sync.length) {
      this.fireSyncDone(syncState);
      return ;
    }
     
    var dataTable = syncState.dataTables2Sync[syncState.i] ;
    try {
      this.syncDataTable(syncState, dataTable);
    }
    catch(e) {
      this.fireSyncFailed(syncState, dataTable, e);
    }
  } ;
  
/**
 * Synchronize a data table
 *
 */
  Sync.prototype.syncDataTable = function(syncState, dataTable) {
  
    var dt = {
     "name"             : dataTable.name, 
     "columnsRequested" : dataTable.columns.slice(), 
     "columns2Send"     : dataTable.columns.slice(), 
     "rowIdColumn"      : _ROWID_COL,
     "mode"             : _MODE_OVERWRITE,
     "sync"             : true
    };
    
    // Check that the data table exists
    var dataTables = getDataTables("!!_!!").split("!!_!!");
    if(dataTables.indexOf(dt.name) === -1) {
      throw "Data Table "+dt.name+" was not found in current document";
    }
    
    // Check synchonization status if we don't want to force
    // the current data table to be sent over
    if(! dataTable.force) {
      var syncStatus    = dng.tsc.getSyncStatus(this.sessionID, dt.name, dt.columnsRequested) ;
      dt.sync           = syncStatus.sync;
      
      if(syncStatus.columnsMissing.length > 0) {
        dt.columnsMissing = syncStatus.columnsMissing.slice();
        dt.columns2Send   = syncStatus.columnsMissing.slice();
        dt.mode           = _MODE_MERGE;
      }
    }

    // First, identify columns that are missing (ignore rowid)
    var columns = getColumns(dt.name, _COLUMNS_SEPARATOR).split(_COLUMNS_SEPARATOR);
    
    var invalidColumns = dt.columns2Send.filter(function(c) {
      return columns.indexOf(c) === -1 && c !== dt.rowIdColumn ;
    });
    
    // If there are columns that were required to be sent but that are
    // missing, throw an error
    if(invalidColumns.length > 0) {
      throw "Column(s) requested are missing in data table "+dt.name+": "+invalidColumns;
    }
    
    // Local status tracking for various sync processes
    dt.status = {
      "tableRowsSent"     : -1,
      "markedRowsSent"    : -1,
      "filteredRowsSent"  : -1,
      "tableProcessed"    : ! dt.sync,
      "markedProcessed"   : ! this.params.syncMarked,
      "filteredProcessed" : ! this.params.syncFiltered,
      "error"             : ''
    };
    
    // Stop here if we don't have to sync the data table at all
    if(! dt.sync && ! this.params.syncMarked && ! this.params.syncFiltered) {
      this.fireSyncNextDone(syncState, dt);
      
      return this ;
    } 
    
    // Recalculate rowidentifier if we are in Overwrite mode
    if(dt.mode === _MODE_OVERWRITE) {
      dng.tsc.computeRowId(dt.name, dt.rowIdColumn);
	    dng.tsc.hideColumn(dt.rowIdColumn);
    }
    else if (dt.mode === _MODE_MERGE) {
      // Merge mode -> Make sure the identifier column is available!
      var columns = getColumns(dt.name, _COLUMNS_SEPARATOR).split(_COLUMNS_SEPARATOR) ;

      if(columns.indexOf(dt.rowIdColumn) < 0) {
        // The identifier column is missing...n Overwrite mode
        dng.tsc.computeRowId(dt.name, dt.rowIdColumn);
        dng.tsc.hideColumn(dt.rowIdColumn);
        dt.mode = _MODE_OVERWRITE ;
      }
    }
    
    // Make sure we also send the row identifier column
    if(dt.columns2Send.indexOf(dt.rowIdColumn) < 0) {
      dt.columns2Send.push(dt.rowIdColumn);
    }
    
    var that = this ;
    
    // Local callback called each time a sync process ends
    var onSyncProcessEnd = function () {
      if(dt.status.tableProcessed && dt.status.markedProcessed && dt.status.filteredProcessed) {
        
        if(dt.status.error != '') {
          that.fireSyncFailed(syncState, dt, dt.status.error);
        }
        else {
          if(dt.sync) {
            // Update columns list depending on the merge status
            var cols = dt.columns2Send ;
            if(dt.mode === _MODE_MERGE) {
              cols = cols
                .concat(dng.tsc.getLastSyncColumns(that.sessionID, dt.name))
                .filter(function(elem, index, self) {
                  return index == self.indexOf(elem);
                });
            }
            
            dng.tsc.setSyncStatus(that.sessionID, dt.name, dt.status.tableRowsSent, cols);
          }
          that.fireSyncNextDone(syncState, dt);
        }
      }
    } ;
    
    this.fireSyncNextStarted(syncState, dt);
    
    setTimeout(function() {
      if(dt.sync) {
        that.dataSender.syncDataTable(dt.name, dt.columns2Send, dt.mode,
          function(callbackData) {
            dt.status.tableProcessed  = true ;
            dt.status.tableRowsSent   = callbackData["RowCount"] ;
            onSyncProcessEnd();
          },
          function(errorText, dataTable) {
            dt.status.tableProcessed = true ;
            dt.status.error          = errorText ;
            onSyncProcessEnd();
          }
        ) ;
      }
      
      if(that.params.syncFiltered) {
        that.dataSender.syncFilteredRows(dt.name, dt.rowIdColumn,
          function(callbackData) {
            dt.status.filteredProcessed  = true ;
            dt.status.filteredRowsSent   = callbackData["RowCount"] ;
            onSyncProcessEnd(callbackData);
          },
          function(errorText, dataTable) {
            dt.status.filteredProcessed = true ;
            dt.status.error             = errorText ;
            onSyncProcessEnd({error: errorText});
          }
        ) ;
      }
      
      if(that.params.syncMarked) {
        that.dataSender.syncMarkedRows(dt.name, dt.rowIdColumn,
          function(callbackData) {
            dt.status.markedProcessed = true ;
            dt.status.markedRowsSent   = callbackData["RowCount"] ;
            onSyncProcessEnd(callbackData);
          },
          function(errorText, dataTable) {
            dt.status.markedProcessed   = true ;
            dt.status.error             = errorText ;
            onSyncProcessEnd({error: errorText});
          }
        ) ;
      }
    }, 10);
    
    return this;
  } ;
  
	
/**************************************************************************
            ------------ Synchronization methods -------------
***************************************************************************/

  
	my.syncUPLOAD = function(pParams) {
    return new SyncUPLOAD(pParams);
  };
  
  function SyncUPLOAD(pParams) {
    if(typeof pParams === 'undefined' || !pParams || typeof pParams !== 'object') {
      throw "SyncUPLOAD: invalid or missing parameters (expected JSON object)";
    }
    
    this.ppServerRoot    = pParams.ppServerRoot; 
    if(typeof this.ppServerRoot !== 'string' || this.ppServerRoot === '') {
      throw "SyncUPLOAD: invalid or missing parameters ppServerRoot";
    }
    
    this.ppSessionID     = pParams.ppSessionID; 
    if(typeof this.ppSessionID !== 'string' || this.ppSessionID === '') {
      throw "SyncUPLOAD: invalid or missing parameters ppSessionID";
    }
    
    this.ppDataDirectory = pParams.ppDataDirectory; 
    if(typeof this.ppDataDirectory !== 'string' || this.ppDataDirectory === '') {
      throw "SyncUPLOAD: invalid or missing parameters ppDataDirectory";
    }
    this.tscSessionID    = pParams.tscSessionID;
    if(typeof this.tscSessionID !== 'string' || this.tscSessionID === '') {
      throw "SyncUPLOAD: invalid or missing parameters tscSessionID";
    }
    
    this.ppProtocolFunctionAll      = pParams.ppProtocolFunctionAll;
    this.ppProtocolFunctionMarked   = pParams.ppProtocolFunctionMarked; 
    this.ppProtocolFunctionFiltered = pParams.ppProtocolFunctionFiltered; 
  };

  SyncUPLOAD.prototype.syncDataTable = function(dataTableName, columns, mode, onDone, onFail) {
    var sbdfSource 	= this.ppDataDirectory+"/"+this.tscSessionID+".sbdf" ;
    // Export data using direct upload
    uploadAllData(
      dataTableName,
      this.ppServerRoot,  
      this.ppSessionID, 
      this.ppDataDirectory, 
      this.tscSessionID, 
      columns
    );
      
    // Run PP Protocol to store data table in PP server
    this.ppProtocolFunctionAll.call(this, {
        "Session Identifier" : this.tscSessionID, 
        "Data Table Name"    : dataTableName, 
        "SBDF Source"        : sbdfSource, 
        "Mode"               : mode
      },
      onDone, 
      {}, 
      onFail
    ) ;
  };
  
  SyncUPLOAD.prototype.syncMarkedRows = function(dataTableName, rowId, onDone, onFail) {
    var sbdfSource 	= this.ppDataDirectory+"/"+this.tscSessionID+"_marked.sbdf" ;
    uploadMarkedData(
      dataTableName, 
      this.ppServerRoot, 
      this.ppSessionID, 
      this.ppDataDirectory, 
      this.tscSessionID+'_marked', 
      'SBDF', 
      '', 
      rowId
    );
    // Call AEP protocol synchronizing marked rows function in ajax on the current data table
    this.ppProtocolFunctionMarked.call(this, {
        "Session Identifier" : this.tscSessionID, 
        "Data Table Name"    : dataTableName, 
        "SBDF Source"        : sbdfSource
      },
      onDone, 
      {}, 
      onFail
    ) ;
  };

  SyncUPLOAD.prototype.syncFilteredRows = function(dataTableName, rowId, onDone, onFail) {
    var sbdfSource 	= this.ppDataDirectory+"/"+this.tscSessionID+"_filtered.sbdf" ;
    
    uploadFilteredData(
      dataTableName, 
      this.ppServerRoot, 
      this.ppSessionID, 
      this.ppDataDirectory, 
      this.tscSessionID+'_filtered', 
      'SBDF', 
      rowId
    );
    
    // Call AEP protocol synchronizing marked rows function in ajax on the current data table
    this.ppProtocolFunctionFiltered.call(this, {
        "Session Identifier" : this.tscSessionID, 
        "Data Table Name"    : dataTableName, 
        "SBDF Source"        : sbdfSource
      },
      onDone, 
      {}, 
      onFail
    ) ;
  };

/**************************************************************************
            ------------ Global synchronization methods -------------
***************************************************************************/

/**  
 * Checks whether a given data table requires synchronization with Pipeline Pilot.
 *  
 *
 */  
	my.getSyncStatus = function(pSessionID, pDataTableName, pColumns) {
		// Get all columns if none is provided
		if(typeof pColumns === "undefined" || !pColumns || pColumns.length == 0) {
			pColumns = getColumns(pDataTableName, _COLUMNS_SEPARATOR).split(_COLUMNS_SEPARATOR);
		}
		else {
		// Check that columns is an array
			if(pColumns.constructor !== Array) {
				// OK, not an array, throw an error
				throw new TypeError("dataTableNeedsSync: columns variable is not an array");
			}
		}
		var n0 = this.getLastSyncRowCount(pSessionID, pDataTableName) ;
		var nc = this.getDataTableRowCount(pDataTableName) ;
		if(n0) {
			// Sync if the # of rows has changed
			if(n0 !== nc) {
				return {"sync": true, "numRows": nc, "columnsMissing": []};
			}
			
			// Check columns now: we want to see if the columns that need to be sync 
			// have alreay been sent during the last sync process
			var lastSyncColumns = this.getLastSyncColumns(pSessionID, pDataTableName);
			var columnsMissing  = [];
      pColumns.forEach(function(c) {
        var found = false ;

				for(var j = 0 ; j < lastSyncColumns.length ; j++) {
					if(c === lastSyncColumns[j]) {
						found = true ; break ;
					}
				}
				
				// Sync if a columns was not found
				if(!found) {
					columnsMissing.push(c);
				}
      });
      
			if(columnsMissing.length === 0) {
				return {"sync": false, "numRows": nc, "columnsMissing": []}; 
			}
      
			return {"sync": true, "numRows": nc, "columnsMissing": columnsMissing}; 
		}
		else {
			// Data table has never been sync as we don't have the document property
			// for rowcount, so we need to sync it
			return {"sync": true, "numRows": nc, "columnsMissing": []};
		}

		return {"sync": false, "numRows": nc, "columnsMissing": []} ;
	};
  
	my.setSyncedRowCount = function(pSessionID, pDataTableName, rowCount, updateSession) {
    if(typeof updateSession === "undefined" || updateSession) {
      setDocumentProperty(
        dng.tsc.getPPSIDPropertyName(), 
        pSessionID
      );
    }
		setDocumentProperty(
			dng.tsc.getRowCountPropertyName(pSessionID, pDataTableName), 
			rowCount
		);
	};
	
	my.setSyncedColumns = function(pSessionID, pDataTableName, columns, updateSession) {
  
		if(typeof columns === "undefined" || !columns || columns.length <= 0) {
      throw "Invalid or empty parameters 'columns': expected a non-empty array of column names.";
		}
    
    var sortedColumns = columns.slice();
    sortedColumns.sort();
    
		var columnList = sortedColumns.join(_COLUMNS_SEPARATOR) ;
		
    if(typeof updateSession === "undefined" || updateSession) {
      setDocumentProperty(
        dng.tsc.getPPSIDPropertyName(), 
        pSessionID
      );
    }
    
		setDocumentProperty(
			this.getSyncedColumnsPropertyName(pSessionID, pDataTableName), 
			columnList
		);
	};

	my.setSyncStatus = function(pSessionID, pDataTableName, rowCount, columns) {
		setDocumentProperty(
      dng.tsc.getPPSIDPropertyName(), 
      pSessionID
    );
    
		dng.tsc.setSyncedColumns(pSessionID, pDataTableName, columns, false);
		dng.tsc.setSyncedRowCount(pSessionID, pDataTableName, rowCount, false);
	};
  
	my.getPPSIDPropertyName = function() {
		return 'PPSID';
	};
  
	my.getRowCountPropertyName = function(pSessionID, pDataTableName) {
		return 'PPSYNC'+("N"+dng.tsc.hashCode(pDataTableName+'_'+pSessionID)).replace("-", "m") ;
	};
	
	my.getSyncedColumnsPropertyName = function(pSessionID, pDataTableName) {
		return 'PPSYNC'+("C"+dng.tsc.hashCode(pDataTableName+'_'+pSessionID)).replace("-", "m") ;
	};
	
	my.getLastSyncColumns = function(pSessionID, pDataTableName) {
		var c = getDocumentProperty(
      this.getSyncedColumnsPropertyName(pSessionID, pDataTableName)
    );
    
		if(c && c !== '') {
			c = c.split(_COLUMNS_SEPARATOR);
		}
    
		return c ;
	};
	
	my.getLastSyncRowCount = function(pSessionID, pDataTableName) {
		return getDocumentProperty(
      dng.tsc.getRowCountPropertyName(pSessionID, pDataTableName)
     );
  }
  
/**************************************************************************
            ------------ Utlities -------------
***************************************************************************/

	my.hashCode = function(str) {
		var hash = 0, 
			len = str.length,
				i, 
			chr;
		
		if (len == 0) return hash;
		
		for (i = 0 ; i < len; i++) {
			hash  = ((hash << 5) - hash) + str.charCodeAt(i);
			hash |= 0; // Convert to 32bit integer
		}
		
		return hash;
	};

/**
 * Utility function that can be used to initialize the list of data tables
 * and put the resulting list into a select box item.
 *
 * @param *selectBoxId the DOM identifier of the target select box
 * @param textFieldId the optional DOM identifier of the associated free-text box
 * @param activeIsDefault whether the active data table should be selected by default
 * @param filter an optional filter function to further refine the list of data tables
 *
 */
	my.initDataTables = function(selectBoxId, textFieldId, activeIsDefault, filter) {

		var select = document.getElementById(selectBoxId);
		
		activeIsDefault = typeof activeIsDefault !== 'undefined' ? activeIsDefault : true ;
		
		//Empty the list box
		select.options.length = 0;
		
		if(textFieldId) {
			select.options[select.options.length] = new Option('- New -', '- New -');
		}
		
		var str 	 = getDataTables(';');
		var activeDT = '' ;
		if(str) {
			var SpotDataTables = str.split(';');
			
			//Set list box content with Spotfire column names
			for(index in SpotDataTables) {
				select.options[select.options.length] = new Option(SpotDataTables[index], SpotDataTables[index]);
			}
			
			activeDT = getActiveDataTable() ;
			if(activeDT && activeDT != '' && activeIsDefault) {
				select.value = activeDT ;
			}
			else if(textFieldId){
				select.value = '- New -' ;
			}
			else {
				select.value = SpotDataTables[0] ;
			}
		}
		
		if(textFieldId) {
			var tf = document.getElementById(textFieldId);
			if(select.value === '- New -') { 
				tf.style.display = 'inline-block';
			}
			else {
				tf.style.display = 'none';
			}
		}
	};
  
/**
 * Utility function that can be used to hide a given column from all Table
 * visualizations available in the current Spotfire document.
 *
 * If the column is not available, nothing happens.
 *
 * @param *columnName the name of the column to hide
 */
	my.hideColumn = function(columnName) {
		var script = 
'from Spotfire.Dxp.Data import *\n\
from Spotfire.Dxp.Data.Import import *\n\
from Spotfire.Dxp.Application import Page\n\
from Spotfire.Dxp.Application.Visuals import *\n\
\n\
for p in Document.Pages: \n\
	for v in p.Visuals:\n\
		if v.TypeId.Name == "Spotfire.Table": \n\
			chart = v.As[TablePlot]()\n\
			(found, column) = chart.Data.DataTableReference.Columns.TryGetValue("'+columnName+'")\n\
			if found:\n\
				chart.TableColumns.Remove(column)\n\
\n\
';
		runScript(base64.encode(script)) ; 
	};
  
/**
 * Utility function that can be used to add a new calculated column storing
 * row identifiers. The internal Spotfire RowId() function is used for this purpose.
 *
 * If a column already exists with the same name, it will be overwritten.
 *
 * @param *dataTableName the name of the target data table
 * @param *pRowIdColumnName the name of the column that will contain the row identifier
 */
	my.computeRowId = function(dataTableName, pRowIdColumnName) {
    
		var script =
'from Spotfire.Dxp.Data import *\n\
from Spotfire.Dxp.Data.Import import *\n\
\n\
dm = Document.Data\n\
dt = dm.Tables["'+dataTableName+'"]\n\
tp = dt.Properties\n\
\n\
columns = dt.Columns\n\
\n\
if not columns.IsValidName ("'+pRowIdColumnName+'"):\n\
\tcolumns.Remove("'+pRowIdColumnName+'")\n\
\n\
newCol = columns.AddCalculatedColumn("'+pRowIdColumnName+'", "RowId()").As[CalculatedColumn]()\n\
newCol.Freeze()\n\
'; 
		runScript(base64.encode(script));
		
		return script ;
	};
  
/**
 * Utility function that can be used to retrieve the number of rows in a
 * Spotfire data table.
 *
 * @param *dataTableName the name of the target data table
 */
	my.getDataTableRowCount = function(dataTableName) {

		// Generate the IronPython script
		var script = "\n\
from Spotfire.Dxp.Data import * \n\
from Spotfire.Dxp.Data.Import import * \n\
\n\
dt = Document.Data.Tables[\""+dataTableName+"\"] \n\
\n\
(exists, prop) = Document.Data.Properties.TryGetProperty(DataPropertyClass.Document, \"TSCTmpROWCOUNT\") \n\
if exists : \n\
	prop.Value = dt.RowCount \n\
else: \n\
	p = DataProperty.CreateCustomPrototype(\"TSCTmpROWCOUNT\", DataType.Integer, DataPropertyAttributes.IsEditable) \n\
	p.Value = dt.RowCount \n\
	Document.Data.Properties.AddProperty(DataPropertyClass.Document, p) \n\
\n\
";
		runScript(base64.encode(script)) ;

		// Get output value
		var count = getDocumentProperty("TSCTmpROWCOUNT");
		script = "\n\
from Spotfire.Dxp.Data import * \n\
from Spotfire.Dxp.Data.Import import * \n\
\n\
Document.Data.Properties.RemoveProperty(DataPropertyClass.Document, \"TSCTmpROWCOUNT\") \n\
\n\
";
		runScript(base64.encode(script)) ;
		return count ;
	}; 
	
	return my ;
	
}(dng.tsc || {})
) ;

  