var dng = (function(dng) {
    return dng;
})(dng || {});

dng.tsc = (function(my) {
    var _COLUMNS_SEPARATOR = '!!_!!';
    var _SYN_DOC_PROP_PREFIX = 'PPSYNC';
    var _ROWID_COL = 'TSC_ROWID';
    var _MODE_OVERWRITE = 'Overwrite';
    var _MODE_MERGE = 'Merge';

    var _spotfireDocument = null;

    /*************************************************************************
     ------------ Synchronization -------------
     ***************************************************************************/

    my.sync = function(pSessionID, pParams, pSpotfireDocument) {
        if (!pSessionID) throw new Error('Session ID is undefined');

        return new Sync(pSessionID, pParams, pSpotfireDocument);
    };

    function Sync(pSessionID, pParams, pSpotfireDocument) {
        this.sessionID = pSessionID;
        this.dataSender = null;
        this.errorCaught = null;
        _spotfireDocument = pSpotfireDocument;

        pParams = !pParams || typeof pParams !== 'object' ? {} : pParams;

        this.params = {
            syncFiltered: pParams.syncFiltered || false,
            syncMarked: pParams.syncMarked || false,
        };

        this._callbacks = {
            syncStart: [],
            syncNextStart: [],
            syncNextDone: [],
            syncNextFail: [],
            syncDone: [],
            syncFail: [],
        };

        this._callbacksInternal = {
            syncStart: [],
            syncNextStart: [],
            syncNextDone: [],
            syncNextFail: [],
            syncDone: [],
            syncFail: [],
        };

        this._callbacksInternal.syncNextDone.push(function(
            syncState,
            dataTable
        ) {
            syncState.i++;
            syncState.dataTablesProcessed.push(dataTable);
            syncState.dataTablesMap[dataTable.name] = dataTable;

            return this.syncNext(syncState);
        });

        this._callbacksInternal.syncDone.push(function(syncState) {
            syncState.running = false;
        });

        this._callbacksInternal.syncFail.push(function(
            syncState,
            dataTable,
            errorMessage
        ) {
            syncState.running = false;
        });
    }

    /**
     * Prepares the current Spotfire document for synchronization. This involves
     * the following :
     * - Resetting document properties related to synchronization if the session identifier
     *   is different from the current one (pSessionID)
     * - Setting the current session identifier document property value
     */
    Sync.prototype._prepareDocument = function(pSessionID) {
        var that = this;

        // Reset session properties if we have switched to a different session
        return dng.tsc
            .getDocumentPropertyAsync(dng.tsc.getPPSIDPropertyName())
            .then(function(oldSessionID) {
                if (oldSessionID && oldSessionID != pSessionID) {
                    // Reset document properties if a different session is used for the current
                    // document
                    return _spotfireDocument
                        .getPropertiesAsync()
                        .then(function(properties) {
                            properties
                                .filter(function(prop) {
                                    return (
                                        prop.name.indexOf(
                                            _SYN_DOC_PROP_PREFIX
                                        ) == 0
                                    );
                                })
                                .forEach(function(prop) {
                                    _spotfireDocument.editor.setDocumentProperty(
                                        prop.name,
                                        'null'
                                    );
                                });
                            return;
                        });
                }
                return;
            })
            .then(function() {
                _spotfireDocument.editor.setDocumentProperty(
                    dng.tsc.getPPSIDPropertyName(),
                    pSessionID
                );

                // apply all the previous setDocumentProperty
                return _spotfireDocument.editor.applyStateAsync();

            })
            .then(function() {
                return that;
            });
    };

    /**
     * Defines the callback function in charge of exporting the data table (and
     * the filtered / marked rows) and importing it back to the Pipeline Pilot server.
     *
     * The function must have the following functions available:
     *
     * - syncDataTable(dataTable, onDone, onFail)
     * - syncMarkedRows(dataTable, onDone, onFail)
     * - syncFilteredRows(dataTable, onDone, onFail)
     *
     */
    Sync.prototype.setDataSender = function(f) {
        this.dataSender = f;
        // Sanity check for data sender
        if (typeof f.syncDataTable !== 'function')
            throw new Error(
                'Invalid data sender: expected prototype function syncDataTable was not found'
            );

        if (typeof f.syncFilteredRows !== 'function')
            throw new Error(
                'Invalid data sender: expected prototype function syncFilteredRows was not found'
            );

        if (typeof f.syncMarkedRows !== 'function')
            throw new Error(
                'Invalid data sender: expected prototype function syncMarkedRows was not found'
            );

        return this;
    };

    /**
     * Register a callback for the specified synchronization event. The following events
     * are currently supported:
     *
     * onSyncStart     : when the synchronization itself starts
     * onSyncNextStart : when the synchronization of the current data table starts
     * onSyncNextDone  : when the synchronization of the current data table ends
     * onSyncDone      : when the synchronization process is done
     * onSyncFail      : when the synchronization process failed (uncatched exception)
     *
     * Each event will trigger the call to the target callback with the following parameter
     *
     * onSyncStart     : f(sync)
     * onSyncNextStart : f(sync, dataTable)
     * onSyncNextDone  : f(sync, dataTable)
     * onSyncNextFail  : f(sync, dataTable)
     * onSyncDone      : f(sync)
     * onSyncFail      : f(sync, dataTable, errorMessage)
     *
     */
    Sync.prototype.on = function(event, callback) {
        if (!this._callbacks.hasOwnProperty(event))
            throw new Error('Unsupported synchronization event: ' + event);

        this._callbacks[event].push(callback);

        return this;
    };

    /**
     * Fires a onSyncStart event
     *
     */
    Sync.prototype.fireSyncStarted = function(syncState) {
        var event = 'syncStart';
        var that = this;

        this._callbacks[event].forEach(function(c) {
            c.call(that);
        });

        this._callbacksInternal[event].forEach(function(c) {
            c.call(that, syncState);
        });
    };

    /**
     * Fires a onSyncStart event
     *
     */
    Sync.prototype.fireSyncNextStarted = function(syncState, dataTable) {
        var event = 'syncNextStart';
        var that = this;

        this._callbacks[event].forEach(function(c) {
            c.call(that, dataTable);
        });

        this._callbacksInternal[event].forEach(function(c) {
            c.call(that, syncState, dataTable);
        });
    };

    /**
     * Fires a onSyncNextDone event
     *
     */
    Sync.prototype.fireSyncNextDone = function(syncState, dataTable) {
        var event = 'syncNextDone';
        var that = this;

        this._callbacks[event].forEach(function(c) {
            c.call(that, dataTable);
        });

        return Promise.all(
            this._callbacksInternal[event].map(function(c) {
                return c.call(that, syncState, dataTable);
            })
        );
    };

    /**
     * Fires a onSyncDone event
     *
     */
    Sync.prototype.fireSyncDone = function(syncState) {
        var event = 'syncDone';
        var that = this;

        this._callbacks[event].forEach(function(c) {
            c.call(that, syncState);
        });

        this._callbacksInternal[event].forEach(function(c) {
            c.call(that, syncState);
        });
    };

    /**
     * Fires a onSyncFail event
     *
     */
    Sync.prototype.fireSyncFailed = function(
        syncState,
        dataTable,
        errorMessage
    ) {
        var event = 'syncFail';
        var that = this;

        this._callbacks[event].forEach(function(c) {
            c.call(that, syncState, dataTable, errorMessage);
        });

        this._callbacksInternal[event].forEach(function(c) {
            c.call(that, syncState, dataTable, errorMessage);
        });

        this.errorCaught = new Error(errorMessage);
    };

    /**
     * Initializes a new synchronization state
     *
     */
    Sync.prototype._initSyncState = function(pDataTables) {
        return {
            dataTables2Sync: pDataTables.slice(),
            dataTablesProcessed: [],
            dataTablesMap: {},
            running: false,
            i: 0,
        };
    };

    /**
     * Start the synchronization procedure.
     *
     */
    Sync.prototype.start = function(pDataTables, pMode) {
        this.errorCaught = null;
        if (this.running) {
            return Promise.resolve();
        }

        var syncState = this._initSyncState(pDataTables);

        // Populate the list of data tables to synchonize
        this.fireSyncStarted(syncState);

        return this.syncNext(syncState, pMode);
    };

    /**
     * Synchronize the next data table.
     *
     */
    Sync.prototype.syncNext = function(syncState, pMode) {
        if (this.errorCaught) {
            return Promise.reject(this.errorCaught)
        }

        // TO-DO: should we use Promise.all to fire al DT synchronisation in parallel?
        if (syncState.i >= syncState.dataTables2Sync.length) {
            this.fireSyncDone(syncState);
            return Promise.resolve();
        } else {
            var dataTable = syncState.dataTables2Sync[syncState.i];
            var that = this;
            return this.syncDataTable(syncState, dataTable, pMode).catch(function(error) {
                that.fireSyncFailed(syncState, dataTable, error);
            });
        }
    };

    Sync.prototype.onSyncProcessEnd = function(dt, syncState) {
        var that = this;
        if (
            dt.status.tableProcessed &&
            dt.status.markedProcessed &&
            dt.status.filteredProcessed
        ) {
            if (dt.status.error != '') {
                return that.fireSyncFailed(
                    syncState,
                    dt,
                    dt.status.error
                );
            }
            if (dt.sync) {
                // Update columns list depending on the merge status
                var cols = dt.columns2Send;
                if (dt.mode === _MODE_MERGE) {
                    cols = cols
                        .concat(
                            dng.tsc.getLastSyncColumns(
                                that.sessionID,
                                dt.name
                            )
                        )
                        .filter(function(
                            elem,
                            index,
                            self
                        ) {
                            return (
                                index == self.indexOf(elem)
                            );
                        });
                }

                return dng.tsc
                    .setSyncStatus(
                        that.sessionID,
                        dt.name,
                        dt.status.tableRowsSent,
                        cols
                    )
                    .then(function() {
                        return that.fireSyncNextDone(
                            syncState,
                            dt
                        );
                    });
            }
            return that.fireSyncNextDone(syncState, dt);
        }
        return;
    };

    Sync.prototype.syncDataTableIfNeeded = function(dt, syncState) {
        if (!dt.sync) {
            return Promise.resolve();
        }

        var that = this;

        return new Promise(function(resolve, reject) {
            that.dataSender.syncDataTable(
                dt.name,
                dt.columns2Send,
                dt.mode,
                function(callbackData) {
                    dt.status.tableProcessed = true;
                    dt.status.tableRowsSent =
                        callbackData['RowCount'];
                    resolve(that.onSyncProcessEnd(dt, syncState));
                },
                function(errorText, dataTable) {
                    dt.status.tableProcessed = true;
                    dt.status.error = errorText;
                    that.onSyncProcessEnd(dt, syncState);
                    reject(errorText);
                }
            );
        });
    };

    Sync.prototype.syncFilteredRowsIfNeeded = function(dt, syncState) {
        if (!this.params.syncFiltered) {
            return Promise.resolve()
        }

        var that = this;

        return new Promise(function(resolve, reject) {
            that.dataSender.syncFilteredRows(
                dt.name,
                dt.rowIdColumn,
                function(callbackData) {
                    dt.status.filteredProcessed = true;
                    dt.status.filteredRowsSent =
                        callbackData['RowCount'];
                    resolve(that.onSyncProcessEnd(dt, syncState));
                },
                function(errorText, dataTable) {
                    dt.status.filteredProcessed = true;
                    dt.status.error = errorText;
                    that.onSyncProcessEnd(dt, syncState);
                    reject(errorText)
                }
            )
        })
    };

    Sync.prototype.syncMarkedRowsIfNeeded = function(dt, syncState) {
        if (!this.params.syncMarked) {
            return Promise.resolve();
        }

        var that = this;
        return new Promise(function(resolve, reject) {
            that.dataSender.syncMarkedRows(
                dt.name,
                dt.rowIdColumn,
                function (callbackData) {
                    dt.status.markedProcessed = true;
                    dt.status.markedRowsSent =
                        callbackData['RowCount'];
                    resolve(that.onSyncProcessEnd(dt, syncState));
                },
                function (errorText, dataTable) {
                    dt.status.markedProcessed = true;
                    dt.status.error = errorText;
                    that.onSyncProcessEnd(dt, syncState);
                    reject(errorText);
                }
            );
        })
    };

    /**
     * Synchronize a data table
     *
     */
    Sync.prototype.syncDataTable = function(syncState, dataTable, pMode) {
        var that = this;
        var dt = {
            name: dataTable.name,
            columnsRequested: dataTable.columns.slice(),
            columns2Send: dataTable.columns.slice(),
            rowIdColumn: _ROWID_COL,
            mode: pMode === _MODE_OVERWRITE || pMode === _MODE_MERGE ? pMode : _MODE_OVERWRITE,
            sync: true,
        };

        // Check that the data table exists
        return _spotfireDocument
            .spotfireConnector.dataTableExistsAsync(dt.name)
            .then(function(dataTableExists) {
                if (!dataTableExists)
                    throw new Error(
                        'Data Table ' +
                            dt.name +
                            ' was not found in current document'
                    );
            })
            .then(function() {
                return _spotfireDocument.getDataColumnsAsync(dt.name)
            })
            .then(function(columns) {
                dt.columns = columns.slice();
            })
            .then(function() {
                return dng.tsc
                    .getSyncStatus(dt, that.sessionID)
                    .then(function(syncStatus) {
                        dt.sync = syncStatus.sync;

                        if (syncStatus.columnsMissing.length > 0) {
                            dt.columnsMissing = syncStatus.columnsMissing.slice();
                            dt.columns2Send = syncStatus.columnsMissing.slice();
                            dt.mode = _MODE_MERGE;
                        }
                    });
            })
            .then(function() {
                // First, identify columns that are missing (ignore rowid)
                var invalidColumns = dt.columns2Send.filter(function(c) {
                    return dt.columns.indexOf(c) === -1 && c !== dt.rowIdColumn;
                });

                // If there are columns that were required to be sent but that are
                // missing, throw an error
                if (invalidColumns.length > 0) {
                    throw new Error(
                        'Column(s) requested are missing in data table ' +
                            dt.name +
                            ': ' +
                            invalidColumns
                    );
                }

                // Local status tracking for various sync processes
                dt.status = {
                    tableRowsSent: -1,
                    markedRowsSent: -1,
                    filteredRowsSent: -1,
                    tableProcessed: !dt.sync,
                    markedProcessed: !that.params.syncMarked,
                    filteredProcessed: !that.params.syncFiltered,
                    error: '',
                };

                // Stop here if we don't have to sync the data table at all
                if (
                    !dt.sync &&
                    !that.params.syncMarked &&
                    !that.params.syncFiltered
                ) {
                    that.fireSyncNextDone(syncState, dt);
                    return false;
                }

                // Recalculate row identifier if we are in Overwrite mode
                if (dt.mode === _MODE_OVERWRITE) {
                    return dng.tsc
                        .computeRowId(dt.name, dt.rowIdColumn)
                        .then(function() {
                            return dng.tsc.hideColumn(dt.rowIdColumn);
                        })
                        .then(function() {
                            return true;
                        });

                }
                if (dt.mode === _MODE_MERGE) {
                    // Merge mode -> Make sure the identifier column is available!
                    if (dt.columns.indexOf(dt.rowIdColumn) < 0) {
                        // The identifier column is missing...n Overwrite mode
                        return dng.tsc
                            .computeRowId(dt.name, dt.rowIdColumn)
                            .then(function() {
                                return dng.tsc.hideColumn(dt.rowIdColumn);
                            })
                            .then(function() {
                                dt.mode = _MODE_OVERWRITE;
                                return true;
                            });
                    }
                    return true;
                }
            })
            .then(function(proceed) {
                if (proceed) {
                    // Make sure we also send the row identifier column
                    if (dt.columns2Send.indexOf(dt.rowIdColumn) < 0) {
                        dt.columns2Send.push(dt.rowIdColumn);
                    }

                    that.fireSyncNextStarted(syncState, dt);

                    // do not use Promise.all here or in Analyst the sync of the data table and the marked rows will
                    // export marked rows and wait indefinitely
                    return that.syncDataTableIfNeeded(dt, syncState)
                        .then(function() {
                            return that.syncFilteredRowsIfNeeded(dt, syncState);
                        })
                        .then (function() {
                            return that.syncMarkedRowsIfNeeded(dt, syncState);
                        })
                    .then(function() {
                        return that;
                    })
                }

                return Promise.resolve(that);
            });
    };

    Sync.prototype.getActiveDataTable = function() {
        return _spotfireDocument.getActiveDataTableAsync();
    };

    Sync.prototype.getDataTableNames = function() {
        return _spotfireDocument.getDataTableNamesAsync();
    };

    Sync.prototype.getDataColumns = function(dataTableName) {
        return _spotfireDocument.getDataColumnsAsync(dataTableName);
    };

    /**************************************************************************
     ------------ Synchronization methods -------------
     ***************************************************************************/

    my.syncUPLOAD = function(pParams) {
        return new SyncUPLOAD(pParams);
    };

    function SyncUPLOAD(pParams) {
        if (!pParams || typeof pParams !== 'object') {
            throw new Error(
                'SyncUPLOAD: invalid or missing parameters (expected JSON object)'
            );
        }

        this.ppServerRoot = pParams.ppServerRoot;
        if (typeof this.ppServerRoot !== 'string' || this.ppServerRoot === '') {
            throw new Error(
                'SyncUPLOAD: invalid or missing parameters ppServerRoot'
            );
        }

        this.ppSessionID = pParams.ppSessionID;
        if (typeof this.ppSessionID !== 'string' || this.ppSessionID === '') {
            throw new Error(
                'SyncUPLOAD: invalid or missing parameters ppSessionID'
            );
        }

        this.ppDataDirectory = pParams.ppDataDirectory;
        if (
            typeof this.ppDataDirectory !== 'string' ||
            this.ppDataDirectory === ''
        ) {
            throw new Error(
                'SyncUPLOAD: invalid or missing parameters ppDataDirectory'
            );
        }
        this.tscSessionID = pParams.tscSessionID;
        if (typeof this.tscSessionID !== 'string' || this.tscSessionID === '') {
            throw new Error(
                'SyncUPLOAD: invalid or missing parameters tscSessionID'
            );
        }

        this.ppProtocolFunctionAll = pParams.ppProtocolFunctionAll;
        this.ppProtocolFunctionMarked = pParams.ppProtocolFunctionMarked;
        this.ppProtocolFunctionFiltered = pParams.ppProtocolFunctionFiltered;
    }

    SyncUPLOAD.prototype.syncDataTable = function(
        dataTableName,
        columns,
        mode,
        onDone,
        onFail
    ) {
        var sbdfSource =
            this.ppDataDirectory + '/' + this.tscSessionID + '.sbdf';

        var exporterArgs = {
            serverRoot: this.ppServerRoot,
            sessionId: this.ppSessionID,
            destinationPath: this.ppDataDirectory,
            destinationFile: this.tscSessionID + '.sbdf',
            columns: columns,
        };

        var that = this;

        return _spotfireDocument
            .exportTableAsync('PipelinePilot', dataTableName, exporterArgs)
            .then(function() {
                // Run PP Protocol to store data table in PP server
                that.ppProtocolFunctionAll.call(
                    that,
                    {
                        'Session Identifier': that.tscSessionID,
                        'Data Table Name': dataTableName,
                        'SBDF Source': sbdfSource,
                        'Mode': mode,
                    },
                    onDone,
                    {},
                    function(error) {
                        onFail(error);
                    }
                );
            });
    };

    SyncUPLOAD.prototype.syncMarkedRows = function(
        dataTableName,
        rowId,
        onDone,
        onFail
    ) {
        var sbdfSource =
            this.ppDataDirectory + '/' + this.tscSessionID + '_marked.sbdf';

        var exporterArgs = {
            serverRoot: this.ppServerRoot,
            sessionId: this.ppSessionID,
            destinationPath: this.ppDataDirectory,
            destinationFile: this.tscSessionID + '_marked.sbdf',
            columns: [rowId],
            filterRows: {
                type: SpotfireFilterRowType.marking,
            },
        };

        var that = this;

        return _spotfireDocument
            .exportTableAsync('PipelinePilot', dataTableName, exporterArgs)
            .then(function() {
                // Call AEP protocol synchronizing marked rows function in ajax on the current data table
                that.ppProtocolFunctionMarked.call(
                    that,
                    {
                        'Session Identifier': that.tscSessionID,
                        'Data Table Name': dataTableName,
                        'SBDF Source': sbdfSource,
                    },
                    onDone,
                    {},
                    function(error) {
                        onFail(error);
                    }
                );
                return;
            });
    };

    SyncUPLOAD.prototype.syncFilteredRows = function(
        dataTableName,
        rowId,
        onDone,
        onFail
    ) {
        var sbdfSource =
            this.ppDataDirectory + '/' + this.tscSessionID + '_filtered.sbdf';

        var exporterArgs = {
            serverRoot: this.ppServerRoot,
            sessionId: this.ppSessionID,
            destinationPath: this.ppDataDirectory,
            destinationFile: this.tscSessionID + '_filtered.sbdf',
            columns: [rowId],
            filterRows: {
                type: SpotfireFilterRowType.filtering,
            },
        };

        var that = this;

        return _spotfireDocument
            .exportTableAsync('PipelinePilot', dataTableName, exporterArgs)
            .then(function() {
                // Call AEP protocol synchronizing marked rows function in ajax on the current data table
                that.ppProtocolFunctionFiltered.call(
                    that,
                    {
                        'Session Identifier': that.tscSessionID,
                        'Data Table Name': dataTableName,
                        'SBDF Source': sbdfSource,
                    },
                    onDone,
                    {},
                    function(error) {
                        onFail(error);
                    }
                );
                return;
            });
    };

    /**************************************************************************
     ------------ Global synchronization methods -------------
     ***************************************************************************/

    /**
     * Checks whether a given data table requires synchronization with Pipeline Pilot.
     */
    my.getSyncStatus = function(pDataTable, pSessionID) {
        if (!pDataTable || (pDataTable.columnsRequested && !Array.isArray(pDataTable.columnsRequested))) {
            throw new TypeError(
                'getSyncStatus: columnsRequested parameter is not an array'
            );
        }

        var nc = null;
        var that = this;
        var syncStatus = null;

        return _spotfireDocument
            .getDataTableRowsCountAsync(pDataTable.name)
            .then(function(dtRowCount) {
                nc = dtRowCount || 0;
                return that.getLastSyncRowCount(pSessionID, pDataTable.name);
            })
            .then(function(n0) {
                var numRows = parseInt(n0);
                if (!isNaN(numRows)) {
                    // Sync if the # of rows has changed
                    if (n0 != nc) {
                        syncStatus = {
                            sync: true,
                            numRows: nc,
                            columnsMissing: [],
                        };
                        return [];
                    }

                    // Check columns now: we want to see if the columns that need to be sync
                    // have already been sent during the last sync process
                    return that.getLastSyncColumns(pSessionID, pDataTable.name);
                }
                // Data table has never been sync as we don't have the document property
                // for rowcount, so we need to sync it
                syncStatus = { sync: true, numRows: nc, columnsMissing: [] };
                return [];
            })
            .then(function(lastSyncColumns) {
                if (!syncStatus) {
                    var columnsMissing = [];

                    // Get all columns if none is provided
                    var columns = Array.isArray(pDataTable.columnsRequested) && pDataTable.columnsRequested.length > 0
                        ? pDataTable.columnsRequested
                        : pDataTable.columns;
                    columns.forEach(function(c) {
                        var found = false;

                        for (
                            var j = 0;
                            j < lastSyncColumns.length;
                            j++
                        ) {
                            if (c === lastSyncColumns[j]) {
                                found = true;
                                break;
                            }
                        }

                        // Sync if a columns was not found
                        if (!found) {
                            columnsMissing.push(c);
                        }
                    });

                    if (columnsMissing.length === 0)
                        return {
                            sync: false,
                            numRows: nc,
                            columnsMissing: [],
                        };

                    return {
                        sync: true,
                        numRows: nc,
                        columnsMissing: columnsMissing,
                    };
                }

                return syncStatus;
            })
    };

    my.setSyncedRowCount = function(
        pSessionID,
        pDataTableName,
        rowCount,
        shouldUpdateSession,
        pSpotfireDocument
    ) {
        if (pSpotfireDocument) _spotfireDocument = pSpotfireDocument;

        var updateSession = function() {
            if (shouldUpdateSession) {
                return _spotfireDocument.editor
                    .setDocumentProperty(
                        dng.tsc.getPPSIDPropertyName(),
                        pSessionID
                    )
                    .applyStateAsync();
            } else {
                return Promise.resolve();
            }
        };

        return Promise.all([
            updateSession(),
            _spotfireDocument.editor
                .setDocumentProperty(
                    dng.tsc.getRowCountPropertyName(pSessionID, pDataTableName),
                    rowCount
                )
                .applyStateAsync(),
        ]);
    };

    my.setSyncedColumns = function(
        pSessionID,
        pDataTableName,
        columns,
        shouldUpdateSession,
        pSpotfireDocument
    ) {
        if (pSpotfireDocument) _spotfireDocument = pSpotfireDocument;

        if (!columns || columns.length <= 0) {
            throw new Error(
                "Invalid or empty parameters 'columns': expected a non-empty array of column names."
            );
        }

        var sortedColumns = columns.slice();
        sortedColumns.sort();

        var columnList = sortedColumns.join(_COLUMNS_SEPARATOR);

        var updateSession = function() {
            if (shouldUpdateSession) {
                return _spotfireDocument.editor
                    .setDocumentProperty(
                        dng.tsc.getPPSIDPropertyName(),
                        pSessionID
                    )
                    .applyStateAsync();
            } else {
                return Promise.resolve();
            }
        };

        return Promise.all([
            updateSession(),
            _spotfireDocument.editor
                .setDocumentProperty(
                    dng.tsc.getSyncedColumnsPropertyName(
                        pSessionID,
                        pDataTableName
                    ),
                    columnList
                )
                .applyStateAsync(),
        ]);
    };

    my.setSyncStatus = function(pSessionID, pDataTableName, rowCount, columns) {
        return Promise.all([
            _spotfireDocument.editor
                .setDocumentProperty(dng.tsc.getPPSIDPropertyName(), pSessionID)
                .applyStateAsync(),
            dng.tsc.setSyncedColumns(
                pSessionID,
                pDataTableName,
                columns,
                false,
                _spotfireDocument
            ),
            dng.tsc.setSyncedRowCount(
                pSessionID,
                pDataTableName,
                rowCount,
                false,
                _spotfireDocument
            ),
        ]);
    };

    my.getPPSIDPropertyName = function() {
        return 'PPSID';
    };

    my.getRowCountPropertyName = function(pSessionID, pDataTableName) {
        return (
            'PPSYNC' +
            ('N' + dng.tsc.hashCode(pDataTableName + '_' + pSessionID)).replace(
                '-',
                'm'
            )
        );
    };

    my.getSyncedColumnsPropertyName = function(pSessionID, pDataTableName) {
        return (
            'PPSYNC' +
            ('C' + dng.tsc.hashCode(pDataTableName + '_' + pSessionID)).replace(
                '-',
                'm'
            )
        );
    };

    my.getLastSyncColumns = function(pSessionID, pDataTableName) {
        return dng.tsc
            .getDocumentPropertyAsync(
                this.getSyncedColumnsPropertyName(pSessionID, pDataTableName)
            )
            .then(function(prop) {
                if (prop) {
                    prop = prop.split(_COLUMNS_SEPARATOR);
                }
                return prop;
            });
    };

    my.getLastSyncRowCount = function(pSessionID, pDataTableName) {
        return dng.tsc.getDocumentPropertyAsync(
            dng.tsc.getRowCountPropertyName(pSessionID, pDataTableName)
        );
    };

    /**************************************************************************
     ------------ Utilities -------------
     ***************************************************************************/

    my.getDocumentPropertyAsync = function(propName) {
        return _spotfireDocument
            .getPropertiesAsync()
            .then(function(properties) {
                var propertiesNames = properties.map(function(prop) {
                    return prop.name;
                });
                if (propertiesNames.indexOf(propName) > -1)
                    return _spotfireDocument.getPropertyAsync(propName);
                return {};
            });
    };

    my.hashCode = function(str) {
        var hash = 0,
            len = str.length,
            i,
            chr;

        if (len == 0) return hash;

        for (i = 0; i < len; i++) {
            hash = (hash << 5) - hash + str.charCodeAt(i);
            hash |= 0; // Convert to 32bit integer
        }

        return hash;
    };

    /**
     * Utility function that can be used to initialize the list of data tables
     * and put the resulting list into a select box item.
     *
     * @param selectBoxId the DOM identifier of the target select box
     * @param textFieldId the optional DOM identifier of the associated free-text box
     * @param activeIsDefault whether the active data table should be selected by default
     */
    // TO-DO: Move out of RTS ?
    my.initDataTables = function(selectBoxId, textFieldId, activeIsDefault) {
        var select = document.getElementById(selectBoxId);

        activeIsDefault =
            typeof activeIsDefault !== 'undefined' ? activeIsDefault : true;

        //Empty the list box
        select.options.length = 0;

        if (textFieldId) {
            select.options[select.options.length] = new Option(
                '- New -',
                '- New -'
            );
        }

        var SpotDataTables = null;

        //Set list box content with Spotfire data table names
        return _spotfireDocument
            .getDataTableNamesAsync()
            .then(function(dt) {
                SpotDataTables = dt;
                for (var index in SpotDataTables) {
                    select.options[select.options.length] = new Option(
                        SpotDataTables[index],
                        SpotDataTables[index]
                    );
                }
                return;
            })
            .then(function() {
                return _spotfireDocument.getActiveDataTableAsync();
            })
            .then(function(activeDT) {
                if (activeDT && activeIsDefault) {
                    select.value = activeDT;
                } else if (textFieldId) {
                    select.value = '- New -';
                } else {
                    select.value = SpotDataTables[0];
                }

                if (textFieldId) {
                    var tf = document.getElementById(textFieldId);
                    if (select.value === '- New -')
                        tf.style.display = 'inline-block';
                    else tf.style.display = 'none';
                }
                return;
            });
    };

    /**
     * Utility function that can be used to hide a given column from all Table
     * visualizations available in the current Spotfire document.
     *
     * If the column is not available, nothing happens.
     *
     * @param *columnName the name of the column to hide
     */
    my.hideColumn = function(columnName) {
        var script =
            'from Spotfire.Dxp.Data import *\n' +
            'from Spotfire.Dxp.Data.Import import *\n' +
            'from Spotfire.Dxp.Application import Page\n' +
            'from Spotfire.Dxp.Application.Visuals import *\n' +
            'for p in Document.Pages: \n' +
            '    for v in p.Visuals:\n' +
            '        if v.TypeId.Name == "Spotfire.Table": \n' +
            '            chart = v.As[TablePlot]()\n' +
            '            (found, column) = chart.Data.DataTableReference.Columns.TryGetValue("' +
            columnName +
            '")\n' +
            'if found:\n' +
            '    chart.TableColumns.Remove(column)\n';

        return _spotfireDocument.executePythonScriptAsync(script, {});
    };

    /**
     * Utility function that can be used to add a new calculated column storing
     * row identifiers. The internal Spotfire RowId() function is used for this purpose.
     *
     * If a column already exists with the same name, it will be overwritten.
     *
     * @param *dataTableName the name of the target data table
     * @param *pRowIdColumnName the name of the column that will contain the row identifier
     */
    my.computeRowId = function(pDataTableName, pRowIdColumnName) {
        var script = [
            'from Spotfire.Dxp.Data import *',
            'from Spotfire.Dxp.Data.Import import *',
            'from System.Text import *',
            'from System.IO import *',
            'from System import *',
            'import re',
            'dm = Document.Data',
            'table = dm.Tables[jsonParam["dataTableName"]]',
            'columns = table.Columns',
            'if not columns.IsValidName(jsonParam["outputColumnName"]) and not columns.Contains(jsonParam["outputColumnName"]):',
            '  raise ApplicationException("The column " + jsonParam["outputColumnName"] + " is not valid.")',
            'if table.Columns.Contains("_tempRowId"):',
            '  table.Columns.Remove("_tempRowId")',
            'rowIdColumn = table.Columns.AddCalculatedColumn("_tempRowId","RowId()").As[CalculatedColumn]()',
            'rowIdColumn.Freeze()',
            '',
            'guids = StringBuilder()',
            'guids.AppendLine("_tempRowId," + jsonParam["outputColumnName"])',
            '',
            'hasTSC_ROWID = columns.Contains(jsonParam["outputColumnName"])',
            '',
            'if hasTSC_ROWID:',
            '  colCursor = DataValueCursor.CreateFormatted(table.Columns[jsonParam["outputColumnName"]])',
            'else:',
            '  colCursor = DataValueCursor.CreateFormatted(table.Columns["_tempRowId"])',
            '',
            'for row in table.GetRows(colCursor):',
            '  TSC_ROWIDValue = colCursor.CurrentValue',
            '  checkGUID = re.match(r"[0-9a-z]{8}-[0-9a-z]{4}-[0-9a-z]{4}-[0-9a-z]{4}-[0-9a-z]{12}", TSC_ROWIDValue)',
            '  if hasTSC_ROWID and checkGUID is not None:',
            '    guids.AppendLine((row.Index + 1).ToString() + "," + TSC_ROWIDValue)',
            '  else:',
            '    guids.AppendLine((row.Index + 1).ToString() + "," + Guid.NewGuid().ToString())',
            '',
            'stream = MemoryStream(Encoding.UTF8.GetBytes(guids.ToString()))',
            'dataSource = TextFileDataSource(stream)',
            'stream.Close()',
            '',
            'if hasTSC_ROWID:',
            '  table.Columns.Remove(jsonParam["outputColumnName"])',
            '',
            'addColumnsSettings = AddColumnsSettings(table, dataSource, JoinType.LeftOuterJoin)',
            'result = table.AddColumns(dataSource, addColumnsSettings)',
            'table.Columns.Remove("_tempRowId")'
        ].join('\n')

        return _spotfireDocument.executePythonScriptAsync(script, {
            dataTableName: pDataTableName,
            outputColumnName: pRowIdColumnName,
        })
        .then(this.hideColumn(pRowIdColumnName));
    };

    return my;
})(dng.tsc || {});
